package stream

import (
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestCredentials(t *testing.T) {
	global, err := NewAddress(Namespace("test"), Version(1), nil)
	require.NoError(t, err)
	filtered, err := NewAddress(Namespace("test"), Version(1), map[string]string{"k": "v"})
	require.NoError(t, err)
	other, err := NewAddress(Namespace("other"), Version(1), nil)
	require.NoError(t, err)

	t.Run("NoPermissions should reject everything", func(t *testing.T) {
		assert.False(t, NoPermissions().CanListen(global))
		assert.False(t, NoPermissions().CanSend(global))
		assert.Nil(t, NoPermissions().Expires())
	})

	t.Run("AllPermissions should allow everything", func(t *testing.T) {
		assert.True(t, AllPermissions().CanListen(global))
		assert.True(t, AllPermissions().CanSend(global))
		assert.Nil(t, AllPermissions().Expires())
	})

	t.Run("Constructed permissions should report their client ID", func(t *testing.T) {
		creds := NewCredentials("cid", AddressScopes{}, AddressScopes{})
		assert.Equal(t, "cid", creds.ClientID())
		assert.Equal(t, "{id: cid listen: [] send: [] expires: <nil>}", creds.String())
	})

	t.Run("Constructed permissions should follow scoping rules", func(t *testing.T) {
		creds := NewCredentials("cid", AddressScopes{global}, AddressScopes{filtered})
		assert.True(t, creds.CanListen(global))
		assert.True(t, creds.CanListen(filtered))
		assert.False(t, creds.CanListen(other))
		assert.False(t, creds.CanSend(global))
		assert.True(t, creds.CanSend(filtered))
		assert.False(t, creds.CanSend(other))
		assert.Nil(t, creds.Expires())
	})

	t.Run("Expiring permissions should allow matches before expiration", func(t *testing.T) {
		exp := time.Now().Add(time.Hour)
		creds := NewTimedCredentials("cid", AddressScopes{global}, AddressScopes{filtered}, exp)
		assert.True(t, creds.CanListen(global))
		assert.True(t, creds.CanSend(filtered))
		require.NotNil(t, creds.Expires())
		assert.Equal(t, exp, *creds.Expires())
	})

	t.Run("Expiring permissions should reject matches after expiration", func(t *testing.T) {
		exp := time.Now().Add(-time.Hour)
		creds := NewTimedCredentials("cid", AddressScopes{global}, AddressScopes{filtered}, exp)
		assert.False(t, creds.CanListen(global))
		assert.False(t, creds.CanSend(filtered))
		require.NotNil(t, creds.Expires())
		assert.Equal(t, exp, *creds.Expires())
	})

	t.Run("Credentials should marshal and unmarshal successfully", func(t *testing.T) {
		bytes, err := AllPermissions().MarshalBinary()
		require.NoError(t, err)
		assert.Equal(t, []byte{'C', 'r', '3', '\001', '\000', '\000', '*', '\000', '\000', '*', '\000', '\000', '*', '\000', '\000'}, bytes)

		out, err := UnmarshalCredentials(bytes)
		assert.NoError(t, err)
		assert.Equal(t, AllPermissions(), out)
	})

	t.Run("Credentials should marshal with timers", func(t *testing.T) {
		exp := time.Now().AddDate(0, 0, 1)
		creds := NewTimedCredentials("c-123", anyAddressSlice, AddressScopes{}, exp)
		bytes, err := creds.MarshalBinary()
		require.NoError(t, err)

		out, err := UnmarshalCredentials(bytes)
		assert.NoError(t, err)
		assert.Equal(t, creds, out)
	})

	t.Run("Credentials should marshal with empty date", func(t *testing.T) {
		exp := time.Unix(0, 0)
		creds := NewTimedCredentials("c-123", anyAddressSlice, AddressScopes{}, exp)
		bytes, err := creds.MarshalBinary()
		require.NoError(t, err)

		out, err := UnmarshalCredentials(bytes)
		assert.NoError(t, err)
		assert.Equal(t, creds, out)
	})

	t.Run("Credentials should not unmarshal without correct header", func(t *testing.T) {
		creds, err := UnmarshalCredentials([]byte("?"))
		assert.Nil(t, creds)
		assert.Equal(t, ErrInvalidCredentialFormat, err)
	})

	t.Run("Credentials should not unmarshal if truncated", func(t *testing.T) {
		bytes, err := AllPermissions().MarshalBinary()
		require.NoError(t, err)
		assert.Equal(t, []byte{'C', 'r', '3', '\001', '\000', '\000', '*', '\000', '\000', '*', '\000', '\000', '*', '\000', '\000'}, bytes)

		// check all points along string, which hit multiple error checking cases in the code
		for i := 0; i < len(bytes); i++ {
			creds, err := UnmarshalCredentials(bytes[:i])
			assert.Nil(t, creds)
			assert.Equal(t, ErrInvalidCredentialFormat, err)
		}
	})

	t.Run("Credentials should not unmarshal if truncated mid-element", func(t *testing.T) {
		creds, err := UnmarshalCredentials([]byte("Cr3\001\000\000\001"))
		assert.Nil(t, creds)
		assert.Equal(t, ErrInvalidCredentialFormat, err)
	})

	t.Run("Credentials should report invalid dates", func(t *testing.T) {
		creds := NewCredentials("c-123", anyAddressSlice, AddressScopes{})
		bytes, err := creds.MarshalBinary()
		require.NoError(t, err)

		bytes = append(bytes, 0)
		out, err := UnmarshalCredentials(bytes)
		assert.Equal(t, ErrInvalidCredentialFormat, err)
		assert.Nil(t, out)
	})

	t.Run("Credentials should not unmarshal if listen scopes are invalid", func(t *testing.T) {
		creds, err := UnmarshalCredentials([]byte("Cr3\001\000\000cid\000\000?\000\000*\000\000"))
		assert.Nil(t, creds)
		assert.Equal(t, ErrMissingRequiredVersion, err)
	})

	t.Run("Credentials should not unmarshal if send scopes are invalid", func(t *testing.T) {
		creds, err := UnmarshalCredentials([]byte("Cr3\001\000\000cid\000\000*\000\000?\000\000"))
		assert.Nil(t, creds)
		assert.Equal(t, ErrMissingRequiredVersion, err)
	})
}
