package stream

import (
	"net/http"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"github.com/stretchr/testify/assert"
)

func TestErrInvalidAuthorization(t *testing.T) {
	err := ErrInvalidAuthorization("x")
	assert.EqualError(t, err, "Invalid Authorization: x")
	assert.Equal(t, ErrInvalidAuthorizationCode, errors.GetErrorCode(err))
	assert.Equal(t, http.StatusForbidden, errors.GetHTTPStatusOrDefault(err, http.StatusInternalServerError))
	assert.Equal(t, errors.Details{"reason": "x"}, errors.GetDetails(err))
	assert.Equal(t, "x", err.Reason())
	assert.EqualError(t, err, "Invalid Authorization: x")

	// cover reason resolution from string table
	assert.EqualError(t, ErrInvalidAuthHeaders, "Invalid Authorization: The auth header syntax is incorrect")
}

func TestMissingFilterValue(t *testing.T) {
	err := ErrMissingFilterValue("x")
	assert.EqualError(t, err, "Address filter x must have a value")
	assert.Equal(t, ErrMissingFilterValueCode, errors.GetErrorCode(err))
	assert.Equal(t, http.StatusUnprocessableEntity, errors.GetHTTPStatusOrDefault(err, http.StatusInternalServerError))
	assert.Equal(t, errors.Details{"filter": "x"}, errors.GetDetails(err))
}

func TestErrors(t *testing.T) {
	assert.Equal(t, ErrInvalidNamespace, Errors.Get(ErrInvalidNamespaceCode, errors.Details{}))
	assert.Equal(t, ErrInvalidAuthorization("x"), Errors.Get(ErrInvalidAuthorizationCode, errors.Details{"reason": "x"}))
	assert.Equal(t, ErrMissingFilterValue("x"), Errors.Get(ErrMissingFilterValueCode, errors.Details{"filter": "x"}))
}
