package listener

import (
	"sync"
	"sync/atomic"

	"code.justin.tv/devhub/e2ml/libs/stream"
)

type TestListener interface {
	stream.Listener
	Lost() int
	Received() int
	Closed() int
}

type devNull struct {
	trackers map[stream.AddressKey]*stream.Tracker
	lost     int32
	received int32
	closed   int32
	mutex    sync.Mutex
}

func NewDevNull() TestListener {
	return &devNull{trackers: make(map[stream.AddressKey]*stream.Tracker)}
}

func (d *devNull) Lost() int     { return int(atomic.LoadInt32(&d.lost)) }
func (d *devNull) Received() int { return int(atomic.LoadInt32(&d.received)) }
func (d *devNull) Closed() int   { return int(atomic.LoadInt32(&d.closed)) }

func (d *devNull) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	d.mutex.Lock()
	tracker := d.trackers[addr.Key()]
	d.mutex.Unlock()
	return tracker.Current()
}

func (d *devNull) OnStreamClosed(addr stream.Address, cause error) bool {
	d.mutex.Lock()
	defer d.mutex.Unlock()
	if _, ok := d.trackers[addr.Key()]; ok {
		delete(d.trackers, addr.Key())
		atomic.AddInt32(&d.closed, 1)
		return true
	}
	return false
}

func (d *devNull) OnDataLost(desc stream.MessageDescription) bool {
	if !d.tracker(desc.Address()).Accept(desc) {
		return false
	}
	atomic.AddInt32(&d.lost, 1)
	return true
}

func (d *devNull) OnDataReceived(msg stream.Message) bool {
	if !d.tracker(msg.Address()).Accept(msg) {
		return false
	}
	atomic.AddInt32(&d.received, 1)
	return true
}

func (d *devNull) tracker(addr stream.Address) *stream.Tracker {
	key := addr.Key()
	d.mutex.Lock()
	if tracker, ok := d.trackers[key]; ok {
		d.mutex.Unlock()
		return tracker
	}
	tracker := new(stream.Tracker)
	d.trackers[key] = tracker
	d.mutex.Unlock()
	return tracker
}
