package stream

import (
	"strconv"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestOpaqueBytes(t *testing.T) {
	t.Run("should represent nil/empty value correctly", func(t *testing.T) {
		val := OpaqueBytes(nil)
		assert.True(t, val.IsEmpty())
		assert.Empty(t, val.String())

		bytes, err := val.MarshalText()
		assert.NoError(t, err)
		assert.Empty(t, bytes)

		bytes, err = val.MarshalBinary()
		assert.NoError(t, err)
		assert.Empty(t, bytes)

		bytes, err = val.MarshalJSON()
		assert.NoError(t, err)
		assert.Equal(t, []byte(`""`), bytes)
	})

	t.Run("should protect set value correctly", func(t *testing.T) {
		val := OpaqueBytes("value")
		assert.False(t, val.IsEmpty())
		assert.Equal(t, protectedReplacement, val.String())

		bytes, err := val.MarshalText()
		assert.NoError(t, err)
		assert.Equal(t, protectedReplacement, string(bytes))

		bytes, err = val.MarshalBinary()
		assert.NoError(t, err)
		assert.Equal(t, protectedReplacement, string(bytes))

		bytes, err = val.MarshalJSON()
		assert.NoError(t, err)
		assert.Equal(t, strconv.Quote(protectedReplacement), string(bytes))
	})

	t.Run("should accept empty values", func(t *testing.T) {
		var val OpaqueBytes
		raw := []byte("")
		assert.NoError(t, val.UnmarshalBinary(raw))
		assert.True(t, val.IsEmpty())

		assert.NoError(t, val.UnmarshalText(raw))
		assert.True(t, val.IsEmpty())

		assert.Error(t, val.UnmarshalJSON(raw))

		quoted := []byte(`""`)
		assert.NoError(t, val.UnmarshalJSON(quoted))
		assert.True(t, val.IsEmpty())
	})

	t.Run("should allow set from unmarshal", func(t *testing.T) {
		var val OpaqueBytes
		raw := []byte("1234567890")
		assert.NoError(t, val.UnmarshalBinary(raw))
		assert.Equal(t, string(raw), string(val))

		val = nil
		assert.NoError(t, val.UnmarshalText(raw))
		assert.Equal(t, string(raw), string(val))

		assert.Error(t, val.UnmarshalJSON(raw))

		val = nil
		quoted := []byte(`"1234567890"`)
		assert.NoError(t, val.UnmarshalJSON(quoted))
		assert.Equal(t, string(raw), string(val))
	})
}
