package protocol

import (
	"net/http"
	"testing"
	"time"

	"code.justin.tv/devhub/e2ml/libs/errors"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"github.com/stretchr/testify/assert"
)

func TestErrInvalidLength(t *testing.T) {
	err := ErrInvalidLength(4, 0)
	assert.EqualError(t, err, "Invalid length (0 < 4)")
	assert.Equal(t, http.StatusBadRequest, errors.GetHTTPStatusOrDefault(err, http.StatusInternalServerError))
	assert.Equal(t, ErrInvalidLengthCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{"expected": 4, "actual": 0}, errors.GetDetails(err))
}

func TestErrInvalidPayload(t *testing.T) {
	err := ErrInvalidPayload(Sent)
	assert.EqualError(t, err, "Invalid payload for OpCode 'sent'")
	assert.Equal(t, http.StatusBadRequest, errors.GetHTTPStatusOrDefault(err, http.StatusInternalServerError))
	assert.Equal(t, ErrInvalidPayloadCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{"code": int(Sent)}, errors.GetDetails(err))
}

func TestErrRateLimited(t *testing.T) {
	err := ErrRateLimited(10, 10*time.Second)
	assert.EqualError(t, err, "Rate limit of 10 writes over 10s exceeded")
	assert.Equal(t, http.StatusBadRequest, errors.GetHTTPStatusOrDefault(err, http.StatusInternalServerError))
	assert.Equal(t, ErrRateLimitedCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{"count": 10, "period": (10 * time.Second).String()}, errors.GetDetails(err))
}

func TestErrors(t *testing.T) {
	assert.Equal(t, stream.ErrInvalidNamespace, Errors.Get(stream.ErrInvalidNamespaceCode, errors.Details{}))
	assert.Equal(t, ErrEOF, Errors.Get(ErrEOFCode, errors.Details{}))
	assert.Equal(t, ErrInvalidLength(10, 20), Errors.Get(ErrInvalidLengthCode, errors.Details{"expected": 10, "actual": 20}))
	assert.Equal(t, ErrInvalidPayload(Sent), Errors.Get(ErrInvalidPayloadCode, errors.Details{"code": int(Sent)}))
	assert.Equal(t, ErrRateLimited(5, time.Second), Errors.Get(ErrRateLimitedCode, errors.Details{"count": 5, "period": time.Second.String()}))
}
