package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	ackLength          = 14
	ackRequestIDOffset = headerLength
	ackSourceIDOffset  = ackRequestIDOffset + 2
	ackPositionOffset  = ackSourceIDOffset + 4
)

type Ack interface {
	protocol.Message
	ForRequestID() protocol.RequestID
	Source() stream.SourceID
	Position() stream.Position
}

type ackMessage struct {
	forId protocol.RequestID
	src   stream.SourceID
	pos   stream.Position
}

// NewAck creates an Ack
func NewAck(id protocol.RequestID, src stream.SourceID, pos stream.Position) (Ack, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	return &ackMessage{id, src, pos}, nil
}

func (*ackMessage) OpCode() protocol.OpCode            { return protocol.Ack }
func (a *ackMessage) ForRequestID() protocol.RequestID { return a.forId }
func (a *ackMessage) Source() stream.SourceID          { return a.src }
func (a *ackMessage) Position() stream.Position        { return a.pos }
func (a *ackMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, ackLength)
	injectHeader(bytes, ver, a.OpCode())
	binary.BigEndian.PutUint16(bytes[ackRequestIDOffset:], uint16(a.forId))
	binary.BigEndian.PutUint32(bytes[ackSourceIDOffset:], uint32(a.src))
	binary.BigEndian.PutUint32(bytes[ackPositionOffset:], uint32(a.pos))
	return bytes, nil
}

func (a *ackMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) != ackLength {
		return protocol.ErrInvalidLength(ackLength, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[ackRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	a.forId = id
	a.src = stream.SourceID(binary.BigEndian.Uint32(bytes[ackSourceIDOffset:]))
	a.pos = stream.Position(binary.BigEndian.Uint32(bytes[ackPositionOffset:]))
	return nil
}

func (a *ackMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, sourceID=%d, pos=%d", a.OpCode(), a.ForRequestID(), a.Source(), a.Position())
}
