package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAck(t *testing.T) {
	src := stream.None + 3
	pos := stream.Origin + 4
	t.Run("should report values correctly", func(t *testing.T) {
		ack, err := NewAck(protocol.FirstRequest, src, pos)
		require.NoError(t, err)
		assert.Equal(t, protocol.Ack, ack.OpCode())
		assert.Equal(t, protocol.FirstRequest, ack.ForRequestID())
		assert.Equal(t, src, ack.Source())
		assert.Equal(t, pos, ack.Position())
		assert.Equal(t, "<ack> reqID=1, sourceID=3, pos=4", ack.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		ack, err := NewAck(protocol.FirstRequest, src, pos)
		require.NoError(t, err)
		bytes, err := ack.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, ack, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewAck(protocol.NoRequest, src, pos)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := createBlank(protocol.Ack).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Ack).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Ack).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Ack).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(ackLength, 0),
			createBlank(protocol.Ack).Unmarshal(protocol.Current, []byte{}))
	})
}
