package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	closedSourceIDOffset = headerLength
	closedPositionOffset = closedSourceIDOffset + 4
	closedAddressOffset  = closedPositionOffset + 4
)

type Closed interface {
	protocol.Message
	Address() stream.Address
	Source() stream.SourceID
	Position() stream.Position
}

type closedMessage struct {
	addr stream.Address
	src  stream.SourceID
	pos  stream.Position
}

func NewClosed(addr stream.Address, src stream.SourceID, pos stream.Position) (Closed, error) {
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &closedMessage{addr, src, pos}, nil
}

func (*closedMessage) OpCode() protocol.OpCode     { return protocol.Closed }
func (c *closedMessage) Address() stream.Address   { return c.addr }
func (c *closedMessage) Source() stream.SourceID   { return c.src }
func (c *closedMessage) Position() stream.Position { return c.pos }

func (c *closedMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if c.addr != nil {
		key = c.addr.Key()
	}
	bytes := make([]byte, closedAddressOffset+len(key))
	injectHeader(bytes, ver, c.OpCode())
	binary.BigEndian.PutUint32(bytes[closedSourceIDOffset:], uint32(c.src))
	binary.BigEndian.PutUint32(bytes[closedPositionOffset:], uint32(c.pos))
	copy(bytes[closedAddressOffset:], key)
	return bytes, nil
}

func (c *closedMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < closedAddressOffset {
		return protocol.ErrInvalidLength(closedAddressOffset, len(bytes))
	}
	addr, err := stream.ParseAddress(string(bytes[closedAddressOffset:]))
	if err != nil {
		return err
	}
	c.pos = stream.Position(binary.BigEndian.Uint32(bytes[closedPositionOffset:]))
	c.src = stream.SourceID(binary.BigEndian.Uint32(bytes[closedSourceIDOffset:]))
	c.addr = addr
	return nil
}

func (c *closedMessage) String() string {
	return fmt.Sprintf("<%s> addr=%s, sourceID=%d, pos=%d", c.OpCode(), c.Address().Key(), c.Source(), c.Position())
}
