package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestClosed(t *testing.T) {
	src := stream.None + 3
	pos := (stream.Origin + 1)
	addr, err := stream.NewAddress(stream.Namespace("test"), stream.Version(1), nil)
	require.NoError(t, err)

	t.Run("should report values correctly", func(t *testing.T) {
		closed, err := NewClosed(addr, src, pos)
		require.NoError(t, err)
		assert.Equal(t, protocol.Closed, closed.OpCode())
		assert.Equal(t, addr, closed.Address())
		assert.Equal(t, pos, closed.Position())
		assert.Equal(t, "<closed> addr=test@1, sourceID=3, pos=1", closed.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		closed, err := NewClosed(addr, src, pos)
		require.NoError(t, err)
		bytes, err := closed.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, closed, out)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Closed).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Closed).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewClosed(nil, src, pos)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&closedMessage{&badAddr{}, src, pos}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Closed).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(closedAddressOffset, 0),
			createBlank(protocol.Closed).Unmarshal(protocol.Current, []byte{}))
	})
}
