package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestInit(t *testing.T) {
	t.Run("should report values correctly", func(t *testing.T) {
		init, err := NewInit(protocol.FirstRequest, protocol.Current, stream.Reservation, OpaqueBytes("ticket"))
		require.NoError(t, err)
		assert.Equal(t, protocol.Init, init.OpCode())
		assert.Equal(t, protocol.FirstRequest, init.RequestID())
		assert.Equal(t, protocol.Current, init.Version())
		assert.Equal(t, stream.Reservation, init.AuthMethod())
		assert.Equal(t, OpaqueBytes("ticket"), init.AccessCode())
		assert.Equal(t, "<init> version=5, authMethod=reservation, len(code)=6", init.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		init, err := NewInit(protocol.FirstRequest, protocol.Current, stream.Validation, OpaqueBytes("abc.123.def"))
		require.NoError(t, err)
		bytes, err := init.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, init, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewInit(protocol.NoRequest, protocol.Current, stream.Reservation, OpaqueBytes("ticket"))
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := createBlank(protocol.Init).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Init).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := NewInit(protocol.FirstRequest, protocol.Unknown, stream.Reservation, OpaqueBytes("ticket"))
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		_, err = createBlank(protocol.Init).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Init).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report auth method errors", func(t *testing.T) {
		_, err := NewInit(protocol.FirstRequest, protocol.Current, 0, OpaqueBytes("ticket"))
		assert.Equal(t, protocol.ErrInvalidAuthMethod, err)
		bytes, err := (&initMessage{protocol.FirstRequest, protocol.Current, 0, OpaqueBytes("ticket")}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidAuthMethod, createBlank(protocol.Init).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(initAccessCodeOffset, 0),
			createBlank(protocol.Init).Unmarshal(protocol.Current, []byte{}))
	})
}
