package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	joinRequestIDOffset = headerLength
	joinSourceIDOffset  = joinRequestIDOffset + 2
	joinPositionOffset  = joinSourceIDOffset + 4
	joinAddressOffset   = joinPositionOffset + 4
)

// Join attempts to subscribe to a stream address
type Join interface {
	protocol.Request
	Address() stream.Address
	Source() stream.SourceID
	Position() stream.Position
}

type joinMessage struct {
	id   protocol.RequestID
	addr stream.Address
	src  stream.SourceID
	pos  stream.Position
}

// NewJoin creates a Join
func NewJoin(id protocol.RequestID, addr stream.Address, src stream.SourceID, pos stream.Position) (Join, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	return &joinMessage{id, addr, src, pos}, nil
}

func (*joinMessage) OpCode() protocol.OpCode         { return protocol.Join }
func (j *joinMessage) RequestID() protocol.RequestID { return j.id }
func (j *joinMessage) Address() stream.Address       { return j.addr }
func (j *joinMessage) Source() stream.SourceID       { return j.src }
func (j *joinMessage) Position() stream.Position     { return j.pos }

func (j *joinMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if j.addr != nil {
		key = j.addr.Key()
	}
	bytes := make([]byte, joinAddressOffset+len(key))
	injectHeader(bytes, ver, j.OpCode())
	binary.BigEndian.PutUint16(bytes[joinRequestIDOffset:], uint16(j.id))
	binary.BigEndian.PutUint32(bytes[joinSourceIDOffset:], uint32(j.src))
	binary.BigEndian.PutUint32(bytes[joinPositionOffset:], uint32(j.pos))
	copy(bytes[joinAddressOffset:], key)
	return bytes, nil
}

func (j *joinMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	if len(bytes) < joinAddressOffset {
		return protocol.ErrInvalidLength(joinAddressOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[joinRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	addr, err := stream.ParseAddress(string(bytes[joinAddressOffset:]))
	if err != nil {
		return err
	}
	j.id = id
	j.src = stream.SourceID(binary.BigEndian.Uint32(bytes[joinSourceIDOffset:]))
	j.pos = stream.Position(binary.BigEndian.Uint32(bytes[joinPositionOffset:]))
	j.addr = addr
	return nil
}

func (j *joinMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, addr=%s, sourceID=%d, pos=%d", j.OpCode(), j.RequestID(), j.Address().Key(), j.Source(), j.Position())
}
