package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type badAddr struct{ stream.Address }

func (*badAddr) Key() stream.AddressKey { return stream.AddressKey("!!bad") }

func TestJoin(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)
	src := stream.None + 3
	pos := stream.Origin + 2

	t.Run("should report values correctly", func(t *testing.T) {
		join, err := NewJoin(protocol.FirstRequest, addr, src, pos)
		require.NoError(t, err)
		assert.Equal(t, protocol.Join, join.OpCode())
		assert.Equal(t, protocol.FirstRequest, join.RequestID())
		assert.Equal(t, addr, join.Address())
		assert.Equal(t, pos, join.Position())
		assert.Equal(t, "<join> reqID=1, addr=test@1, sourceID=3, pos=2", join.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		join, err := NewJoin(protocol.FirstRequest, addr, src, pos)
		require.NoError(t, err)
		bytes, err := join.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, join, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewJoin(protocol.NoRequest, addr, src, pos)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&joinMessage{protocol.NoRequest, addr, src, pos}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Join).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Join).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Join).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewJoin(protocol.FirstRequest, nil, src, pos)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&joinMessage{protocol.FirstRequest, &badAddr{}, src, pos}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Join).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(joinAddressOffset, 0),
			createBlank(protocol.Join).Unmarshal(protocol.Current, []byte{}))
	})
}
