package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	lostSourceIDOffset = headerLength
	lostStartOffset    = lostSourceIDOffset + 4
	lostEndOffset      = lostStartOffset + 4
	lostAddressOffset  = lostEndOffset + 4
)

type Lost interface {
	stream.MessageDescription
	protocol.Message
}

type lostMessage struct {
	addr stream.Address
	src  stream.SourceID
	seg  stream.Segment
}

func NewLost(addr stream.Address, src stream.SourceID, seg stream.Segment) (Lost, error) {
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	if seg.Distance() == 0 {
		return nil, protocol.ErrInvalidSegment
	}
	return &lostMessage{addr, src, seg}, nil
}

func (*lostMessage) OpCode() protocol.OpCode   { return protocol.Lost }
func (l *lostMessage) Address() stream.Address { return l.addr }
func (l *lostMessage) Source() stream.SourceID { return l.src }
func (l *lostMessage) At() stream.Segment      { return l.seg }

func (l *lostMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if l.addr != nil {
		key = l.addr.Key()
	}
	bytes := make([]byte, lostAddressOffset+len(key))
	injectHeader(bytes, ver, l.OpCode())
	binary.BigEndian.PutUint32(bytes[lostSourceIDOffset:], uint32(l.src))
	binary.BigEndian.PutUint32(bytes[lostStartOffset:], uint32(l.seg.Start))
	binary.BigEndian.PutUint32(bytes[lostEndOffset:], uint32(l.seg.End))
	copy(bytes[lostAddressOffset:], key)
	return bytes, nil
}

func (l *lostMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}

	total := len(bytes)
	if total < lostAddressOffset {
		return protocol.ErrInvalidLength(lostAddressOffset, len(bytes))
	}
	addr, err := stream.ParseAddress(string(bytes[lostAddressOffset:]))
	if err != nil {
		return err
	}
	l.src = stream.SourceID(binary.BigEndian.Uint32(bytes[lostSourceIDOffset:]))
	l.seg = stream.Segment{
		Start: stream.Position(binary.BigEndian.Uint32(bytes[lostStartOffset:])),
		End:   stream.Position(binary.BigEndian.Uint32(bytes[lostEndOffset:])),
	}
	l.addr = addr
	return nil
}

func (l *lostMessage) String() string {
	return fmt.Sprintf("<%v> addr=%s, sourceID=%d, at=[%d:%d]",
		l.OpCode(),
		l.Address().Key(),
		l.Source(),
		l.At().Start,
		l.At().End)
}
