package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestLost(t *testing.T) {
	src := stream.None + 3
	seg := (stream.Origin + 1).AsDelta()
	addr, err := stream.NewAddress(stream.Namespace("test"), stream.Version(1), nil)
	require.NoError(t, err)

	t.Run("should report values correctly", func(t *testing.T) {
		lost, err := NewLost(addr, src, seg)
		require.NoError(t, err)
		assert.Equal(t, protocol.Lost, lost.OpCode())
		assert.Equal(t, addr, lost.Address())
		assert.Equal(t, seg, lost.At())
		assert.Equal(t, "<lost> addr=test@1, sourceID=3, at=[0:1]", lost.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		lost, err := NewLost(addr, src, seg)
		require.NoError(t, err)
		bytes, err := lost.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, lost, out)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Lost).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Lost).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewLost(nil, src, seg)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&lostMessage{&badAddr{}, src, seg}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Lost).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report segment errors", func(t *testing.T) {
		_, err := NewLost(addr, stream.None, stream.Empty)
		assert.Equal(t, protocol.ErrInvalidSegment, err)
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(lostAddressOffset, 0),
			createBlank(protocol.Lost).Unmarshal(protocol.Current, []byte{}))
	})
}
