package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestMove(t *testing.T) {
	addr1, err := stream.ParseAddress("ext@1?a=b")
	require.NoError(t, err)
	addr2, err := stream.ParseAddress("loadtest@1?n=0")
	require.NoError(t, err)

	scopes := stream.AddressScopes{addr1, addr2}

	t.Run("should report values correctly", func(t *testing.T) {
		move, err := NewMove(scopes)
		require.NoError(t, err)
		assert.Equal(t, protocol.Move, move.OpCode())
		assert.Equal(t, scopes, move.Scopes())
		assert.Equal(t, "<move> scopes=[^ext@1?a=b ^loadtest@1?n=0]", move.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		move, err := NewMove(scopes)
		require.NoError(t, err)
		bytes, err := move.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, move, out)
	})

	t.Run("should report scope errors", func(t *testing.T) {
		_, err := NewMove(stream.AddressScopes{})
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := createBlank(protocol.Move).Marshal(protocol.Current)
		require.NoError(t, err)
		bytes = append(bytes, '@')
		assert.Equal(t, stream.ErrMissingRequiredNamespace, createBlank(protocol.Move).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Move).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Move).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(moveScopesOffset, 0),
			createBlank(protocol.Move).Unmarshal(protocol.Current, []byte{}))
	})
}
