package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestPart(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		part, err := NewPart(protocol.FirstRequest, addr)
		require.NoError(t, err)
		assert.Equal(t, protocol.Part, part.OpCode())
		assert.Equal(t, protocol.FirstRequest, part.RequestID())
		assert.Equal(t, addr, part.Address())
		assert.Equal(t, "<part> reqID=1, addr=test@1", part.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		part, err := NewPart(protocol.FirstRequest, addr)
		require.NoError(t, err)
		bytes, err := part.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, part, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewPart(protocol.NoRequest, addr)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&partMessage{protocol.NoRequest, addr}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Part).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Part).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Part).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewPart(protocol.FirstRequest, nil)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&partMessage{protocol.FirstRequest, &badAddr{}}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Part).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(partAddressOffset, 0),
			createBlank(protocol.Part).Unmarshal(protocol.Current, []byte{}))
	})
}
