package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	refreshRequestIDOffset  = headerLength
	refreshMethodOffset     = refreshRequestIDOffset + 2
	refreshAccessCodeOffset = refreshMethodOffset + 1
)

// Refresh is used to update an expiring authorization
type Refresh interface {
	protocol.Request
	AccessCode() OpaqueBytes
	AuthMethod() stream.AuthMethod
}

type refreshMessage struct {
	id         protocol.RequestID
	auth       stream.AuthMethod
	accessCode OpaqueBytes
}

// NewRefresh creates an Init
func NewRefresh(id protocol.RequestID, auth stream.AuthMethod, accessCode OpaqueBytes) (Refresh, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if !auth.IsValid() {
		return nil, protocol.ErrInvalidAuthMethod
	}
	return &refreshMessage{id, auth, accessCode}, nil
}

func (*refreshMessage) OpCode() protocol.OpCode         { return protocol.Refresh }
func (r *refreshMessage) RequestID() protocol.RequestID { return r.id }
func (r *refreshMessage) AccessCode() OpaqueBytes       { return r.accessCode }
func (r *refreshMessage) AuthMethod() stream.AuthMethod { return r.auth }
func (r *refreshMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	bytes := make([]byte, refreshAccessCodeOffset+len(r.accessCode))
	injectHeader(bytes, ver, r.OpCode())
	binary.BigEndian.PutUint16(bytes[refreshRequestIDOffset:], uint16(r.id))
	bytes[refreshMethodOffset] = byte(r.auth)
	copy(bytes[refreshAccessCodeOffset:], r.accessCode)
	return bytes, nil
}

func (r *refreshMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}

	if len(bytes) < refreshAccessCodeOffset {
		return protocol.ErrInvalidLength(refreshAccessCodeOffset, len(bytes))
	}

	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[refreshRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	auth := stream.AuthMethod(bytes[refreshMethodOffset])
	if !auth.IsValid() {
		return protocol.ErrInvalidAuthMethod
	}
	r.id = id
	r.auth = auth
	r.accessCode = OpaqueBytes(bytes[refreshAccessCodeOffset:])
	return nil
}

func (r *refreshMessage) String() string {
	return fmt.Sprintf("<%v> authMethod=%v, len(code)=%d", r.OpCode(), r.AuthMethod(), len(r.AccessCode()))
}
