package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestRefresh(t *testing.T) {
	t.Run("should report values correctly", func(t *testing.T) {
		refresh, err := NewRefresh(protocol.FirstRequest, stream.Reservation, OpaqueBytes("ticket"))
		require.NoError(t, err)
		assert.Equal(t, protocol.Refresh, refresh.OpCode())
		assert.Equal(t, protocol.FirstRequest, refresh.RequestID())
		assert.Equal(t, stream.Reservation, refresh.AuthMethod())
		assert.Equal(t, OpaqueBytes("ticket"), refresh.AccessCode())
		assert.Equal(t, "<refresh> authMethod=reservation, len(code)=6", refresh.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		refresh, err := NewRefresh(protocol.FirstRequest, stream.Validation, OpaqueBytes("abc.123.def"))
		require.NoError(t, err)
		bytes, err := refresh.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, refresh, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewRefresh(protocol.NoRequest, stream.Reservation, OpaqueBytes("ticket"))
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := createBlank(protocol.Refresh).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Refresh).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Refresh).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Refresh).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report auth method errors", func(t *testing.T) {
		_, err := NewRefresh(protocol.FirstRequest, 0, OpaqueBytes("ticket"))
		assert.Equal(t, protocol.ErrInvalidAuthMethod, err)
		bytes, err := (&refreshMessage{protocol.FirstRequest, 0, OpaqueBytes("ticket")}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidAuthMethod, createBlank(protocol.Refresh).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(refreshAccessCodeOffset, 0),
			createBlank(protocol.Refresh).Unmarshal(protocol.Current, []byte{}))
	})
}
