package message

import (
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestRelease(t *testing.T) {
	addr, errx := stream.NewAddress(stream.Namespace("test"), stream.Version(1), map[string]string{})
	require.NoError(t, errx)

	t.Run("should report values correctly", func(t *testing.T) {
		release, err := NewRelease(protocol.FirstRequest, addr)
		require.NoError(t, err)
		assert.Equal(t, protocol.Release, release.OpCode())
		assert.Equal(t, protocol.FirstRequest, release.RequestID())
		assert.Equal(t, addr, release.Address())
		assert.Equal(t, "<release> reqID=1, addr=test@1", release.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		release, err := NewRelease(protocol.FirstRequest, addr)
		require.NoError(t, err)
		bytes, err := release.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, release, out)
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewRelease(protocol.NoRequest, addr)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := (&releaseMessage{protocol.NoRequest, addr}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Release).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Release).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Release).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewRelease(protocol.FirstRequest, nil)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&releaseMessage{protocol.FirstRequest, &badAddr{}}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Release).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(releaseAddressOffset, 0),
			createBlank(protocol.Release).Unmarshal(protocol.Current, []byte{}))
	})
}
