package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

const (
	sendRequestIDOffset = headerLength
	sendFlagsOffset     = sendRequestIDOffset + 2
	sendAddressOffset   = sendFlagsOffset + 1
)

type Send interface {
	protocol.Request
	Address() stream.Address
	Content() []byte
	IsDelta() bool
}

type sendMessage struct {
	id    protocol.RequestID
	addr  stream.Address
	bytes []byte
	flags SendFlags
}

func NewSend(id protocol.RequestID, addr stream.Address, bytes []byte, isDelta bool) (Send, error) {
	if !id.IsValid() {
		return nil, protocol.ErrInvalidRequestID
	}
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	flags := Default
	if isDelta {
		flags |= Delta
	}
	return &sendMessage{id, addr, bytes, flags}, nil
}

func (*sendMessage) OpCode() protocol.OpCode         { return protocol.Send }
func (s *sendMessage) RequestID() protocol.RequestID { return s.id }
func (s *sendMessage) Address() stream.Address       { return s.addr }
func (s *sendMessage) Content() []byte               { return s.bytes }
func (s *sendMessage) IsDelta() bool                 { return s.flags.Includes(Delta) }

func (s *sendMessage) Marshal(ver protocol.Version) ([]byte, error) {
	if !ver.IsValid() {
		return nil, protocol.ErrInvalidVersion
	}
	key := stream.NoAddress
	if s.addr != nil {
		key = s.addr.Key()
	}
	bytes := make([]byte, sendAddressOffset+len(key)+1+len(s.bytes))
	injectHeader(bytes, ver, s.OpCode())
	binary.BigEndian.PutUint16(bytes[sendRequestIDOffset:], uint16(s.id))
	bytes[sendFlagsOffset] = byte(s.flags)
	copy(bytes[sendAddressOffset:], key)
	contentOffset := sendAddressOffset + len(key)
	bytes[contentOffset] = '\000'
	copy(bytes[contentOffset+1:], s.bytes)
	return bytes, nil
}

func (s *sendMessage) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}
	total := len(bytes)
	if total < sendAddressOffset {
		return protocol.ErrInvalidLength(sendAddressOffset, len(bytes))
	}
	contentOffset := sendAddressOffset
	for contentOffset < total {
		if bytes[contentOffset] == '\000' {
			break
		}
		contentOffset++
	}
	if contentOffset == total {
		return protocol.ErrEOF
	}
	id := protocol.RequestID(binary.BigEndian.Uint16(bytes[sendRequestIDOffset:]))
	if !id.IsValid() {
		return protocol.ErrInvalidRequestID
	}
	addr, err := stream.ParseAddress(string(bytes[sendAddressOffset:contentOffset]))
	if err != nil {
		return err
	}
	s.id = id
	s.addr = addr
	s.flags = SendFlags(bytes[sendFlagsOffset])
	s.bytes = make([]byte, total-(contentOffset+1))
	copy(s.bytes, bytes[contentOffset+1:])
	return nil
}

func (s *sendMessage) String() string {
	return fmt.Sprintf("<%v> reqID=%d, addr=%s, delta=%v, hexData={%v}", s.OpCode(), s.RequestID(), s.Address().Key(), s.IsDelta(), HexPrintedBytes(s.Content()))
}
