package message

import (
	"fmt"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestSend(t *testing.T) {
	addr, err := stream.NewAddress(stream.Namespace("test"), stream.Version(1), nil)
	require.NoError(t, err)
	content := []byte("hello world")

	t.Run("should report values correctly", func(t *testing.T) {
		send, err := NewSend(protocol.FirstRequest, addr, content, true)
		require.NoError(t, err)
		assert.Equal(t, protocol.FirstRequest, send.RequestID())
		assert.Equal(t, protocol.Send, send.OpCode())
		assert.Equal(t, addr, send.Address())
		assert.Equal(t, content, send.Content())
		assert.True(t, send.IsDelta())
		assert.Equal(t, fmt.Sprintf("<send> reqID=1, addr=test@1, delta=true, hexData={%v}", HexPrintedBytes(content)), send.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		send, err := NewSend(protocol.FirstRequest, addr, content, false)
		require.NoError(t, err)
		bytes, err := send.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, send, out)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Send).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Send).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report request id errors", func(t *testing.T) {
		_, err := NewSend(protocol.NoRequest, addr, content, false)
		assert.Equal(t, protocol.ErrInvalidRequestID, err)
		bytes, err := createBlank(protocol.Send).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, protocol.ErrInvalidRequestID, createBlank(protocol.Send).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewSend(protocol.FirstRequest, nil, content, false)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		bytes, err := (&sendMessage{protocol.FirstRequest, &badAddr{}, content, Default}).Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Send).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(sendAddressOffset, 0),
			createBlank(protocol.Send).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, sendAddressOffset)
		injectHeader(broken, protocol.Current, protocol.Send)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Send).Unmarshal(protocol.Current, broken))
	})
}
