package message

import (
	"encoding/binary"
	"fmt"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol/message/marshal"
)

const (
	sentSourceIDOffset = headerLength
	sentStartOffset    = sentSourceIDOffset + 4
	sentEndOffset      = sentStartOffset + 4
	sentAddressOffset  = sentEndOffset + 4
)

type Sent interface {
	stream.Message
	protocol.Message
}

type sentData struct {
	addr  stream.Address
	src   stream.SourceID
	seg   stream.Segment
	bytes []byte
}

type sentMessage struct {
	*sentData
	v4 marshal.Cache
	v5 marshal.Cache
}

func NewSent(addr stream.Address, src stream.SourceID, seg stream.Segment, bytes []byte) (Sent, error) {
	if addr == nil {
		return nil, protocol.ErrInvalidAddress
	}
	if seg.Distance() == 0 {
		return nil, protocol.ErrInvalidSegment
	}
	if bytes != nil { // duplicate bytes into a buffer we own
		buffer := make([]byte, len(bytes))
		copy(buffer, bytes)
		bytes = buffer
	}
	data := &sentData{addr, src, seg, bytes}
	msg := &sentMessage{sentData: data}
	msg.v4 = marshal.NewOnceCache(msg.marshalForCache, protocol.Four)
	msg.v5 = marshal.NewOnceCache(msg.marshalForCache, protocol.Five)
	return msg, nil
}

func (*sentData) OpCode() protocol.OpCode   { return protocol.Sent }
func (s *sentData) Address() stream.Address { return s.addr }
func (s *sentData) Source() stream.SourceID { return s.src }
func (s *sentData) At() stream.Segment      { return s.seg }
func (s *sentData) Data() []byte            { return s.bytes }

func (s *sentMessage) Marshal(ver protocol.Version) ([]byte, error) {
	switch ver {
	case protocol.Four:
		return s.v4.Get()
	case protocol.Five:
		return s.v5.Get()
	}
	return nil, protocol.ErrInvalidVersion
}

func (s *sentData) marshalForCache(ver protocol.Version) ([]byte, error) {
	key := stream.NoAddress
	if s.addr != nil {
		key = s.addr.Key()
	}
	bytes := make([]byte, sentAddressOffset+len(key)+1+len(s.bytes))
	injectHeader(bytes, ver, s.OpCode())
	binary.BigEndian.PutUint32(bytes[sentSourceIDOffset:], uint32(s.src))
	binary.BigEndian.PutUint32(bytes[sentStartOffset:], uint32(s.seg.Start))
	binary.BigEndian.PutUint32(bytes[sentEndOffset:], uint32(s.seg.End))
	copy(bytes[sentAddressOffset:], key)
	contentOffset := sentAddressOffset + len(key)
	bytes[contentOffset] = '\000'
	copy(bytes[contentOffset+1:], s.bytes)
	return bytes, nil
}

func (s *sentData) Unmarshal(ver protocol.Version, bytes []byte) error {
	if !ver.IsValid() {
		return protocol.ErrInvalidVersion
	}

	total := len(bytes)
	if total < sentAddressOffset {
		return protocol.ErrInvalidLength(sentAddressOffset, len(bytes))
	}
	seg := stream.Segment{
		Start: stream.Position(binary.BigEndian.Uint32(bytes[sentStartOffset:])),
		End:   stream.Position(binary.BigEndian.Uint32(bytes[sentEndOffset:])),
	}
	contentOffset := sentAddressOffset
	for contentOffset < total {
		if bytes[contentOffset] == '\000' {
			break
		}
		contentOffset++
	}
	if contentOffset == total {
		return protocol.ErrEOF
	}
	addr, err := stream.ParseAddress(string(bytes[sentAddressOffset:contentOffset]))
	if err != nil {
		return err
	}
	s.src = stream.SourceID(binary.BigEndian.Uint32(bytes[sentSourceIDOffset:]))
	s.seg = seg
	s.addr = addr
	s.bytes = bytes[contentOffset+1:]
	return nil
}

func (s *sentMessage) String() string {
	return fmt.Sprintf("<%v> addr=%s, sourceID=%d, at=[%d:%d], hexData={%v}",
		s.OpCode(),
		s.Address().Key(),
		s.Source(),
		s.At().Start,
		s.At().End,
		HexPrintedBytes(s.Data()))
}
