package message

import (
	"fmt"
	"testing"

	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol/message/marshal"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestSent(t *testing.T) {
	src := stream.None + 3
	seg := (stream.Origin + 1).AsDelta()
	addr, err := stream.NewAddress(stream.Namespace("test"), stream.Version(1), nil)
	require.NoError(t, err)
	content := []byte("hello world")

	t.Run("should report values correctly", func(t *testing.T) {
		sent, err := NewSent(addr, src, seg, content)
		require.NoError(t, err)
		assert.Equal(t, protocol.Sent, sent.OpCode())
		assert.Equal(t, addr, sent.Address())
		assert.Equal(t, seg, sent.At())
		assert.Equal(t, content, sent.Data())
		assert.Equal(t, fmt.Sprintf("<sent> addr=test@1, sourceID=3, at=[0:1], hexData={%v}", HexPrintedBytes(content)), sent.String())
	})

	t.Run("should marshal correctly", func(t *testing.T) {
		sent, err := NewSent(addr, src, seg, content)
		require.NoError(t, err)
		bytes, err := sent.Marshal(protocol.Current)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, sent.(*sentMessage).sentData, out.(*sentMessage).sentData)
	})

	t.Run("should report version errors", func(t *testing.T) {
		_, err := createBlank(protocol.Sent).Marshal(protocol.Unknown)
		assert.Equal(t, protocol.ErrInvalidVersion, err)
		assert.Equal(t, protocol.ErrInvalidVersion, createBlank(protocol.Sent).Unmarshal(protocol.Unknown, []byte{}))
	})

	t.Run("should report address errors", func(t *testing.T) {
		_, err := NewSent(nil, src, seg, content)
		assert.Equal(t, protocol.ErrInvalidAddress, err)
		msg, err := NewSent(addr, src, seg, content)
		require.NoError(t, err)
		require.IsType(t, &sentMessage{}, msg)
		msg.(*sentMessage).addr = &badAddr{}
		bytes, err := msg.Marshal(protocol.Current)
		require.NoError(t, err)
		assert.Equal(t, stream.ErrMissingRequiredVersion, createBlank(protocol.Sent).Unmarshal(protocol.Current, bytes))
	})

	t.Run("should report segment errors", func(t *testing.T) {
		_, err := NewSent(addr, stream.None, stream.Empty, content)
		assert.Equal(t, protocol.ErrInvalidSegment, err)
	})

	t.Run("should report length errors", func(t *testing.T) {
		assert.Equal(t, protocol.ErrInvalidLength(sentAddressOffset, 0),
			createBlank(protocol.Sent).Unmarshal(protocol.Current, []byte{}))
	})

	t.Run("should handle EOF gracefully", func(t *testing.T) {
		broken := make([]byte, sentAddressOffset)
		injectHeader(broken, protocol.Current, protocol.Sent)
		broken = append(broken, "key"...)
		assert.Equal(t, protocol.ErrEOF, createBlank(protocol.Sent).Unmarshal(protocol.Current, broken))
	})

	t.Run("should marshal v4 correctly", func(t *testing.T) {
		sent, err := NewSent(addr, src, seg, content)
		require.NoError(t, err)
		bytes, err := sent.Marshal(protocol.Four)
		require.NoError(t, err)
		out, err := Unmarshal(bytes)
		require.NoError(t, err)
		assert.Equal(t, sent.(*sentMessage).sentData, out.(*sentMessage).sentData)
	})
}

func BenchmarkNullCache(b *testing.B) {
	addr, err := stream.NewAddress(stream.Namespace("test"), 123, map[string]string{"k": "v", "a": "b"})
	require.NoError(b, err)
	data := make([]byte, 1024000)
	msg, err := NewSent(addr, stream.None, stream.Position(1).AsDelta(), data)
	require.NoError(b, err)
	cast, ok := msg.(*sentMessage)
	require.True(b, ok)
	cast.v4 = marshal.NewNullCache(cast.sentData.marshalForCache, protocol.Unknown)
	for n := 0; n < b.N; n++ {
		_, _ = msg.Marshal(protocol.Four)
	}
}

func BenchmarkOnceCache(b *testing.B) {
	addr, err := stream.NewAddress(stream.Namespace("test"), 123, map[string]string{"k": "v", "a": "b"})
	require.NoError(b, err)
	data := make([]byte, 1024000)
	msg, err := NewSent(addr, stream.None, stream.Position(1).AsDelta(), data)
	require.NoError(b, err)
	cast, ok := msg.(*sentMessage)
	require.True(b, ok)
	cast.v4 = marshal.NewOnceCache(cast.sentData.marshalForCache, protocol.Unknown)
	for n := 0; n < b.N; n++ {
		_, _ = msg.Marshal(protocol.Four)
	}
}
