package protocol

import "strconv"

// Each message header includes a single byte opcode to identity the handler
// that should be processing the payload.
type OpCode byte

const (
	Invalid OpCode = 0 // message could not be parsed

	clientFlag  OpCode = 1 << 6
	serviceFlag OpCode = 1 << 7

	Init    OpCode = clientFlag | 0 // initial handshake including identity assertion
	Join    OpCode = clientFlag | 1
	Part    OpCode = clientFlag | 2
	Send    OpCode = clientFlag | 4
	Release OpCode = clientFlag | 5
	Refresh OpCode = clientFlag | 6 // update authorization when requested

	Drain    OpCode = serviceFlag | 0 // request for client to reroute and disconnect
	Ack      OpCode = serviceFlag | 1 // message received; additional responses may follow
	Closed   OpCode = serviceFlag | 2 // sent when a source has closed an address
	Sent     OpCode = serviceFlag | 3
	Error    OpCode = serviceFlag | 4
	Lost     OpCode = serviceFlag | 5 // sent when a source has dropped a segment
	Expiring OpCode = serviceFlag | 6 // sent when a client must reauthorize
	Move     OpCode = serviceFlag | 7 // request for a client to reroute specific scopes
)

var (
	codeStrings = map[OpCode]string{
		Ack:      "ack",
		Closed:   "closed",
		Drain:    "drain",
		Error:    "error",
		Expiring: "expiring",
		Init:     "init",
		Join:     "join",
		Move:     "move",
		Lost:     "lost",
		Part:     "part",
		Refresh:  "refresh",
		Release:  "release",
		Send:     "send",
		Sent:     "sent",
	}
)

func (o OpCode) IsFromClient() bool  { return (o & clientFlag) != Invalid }
func (o OpCode) IsFromService() bool { return (o & serviceFlag) != Invalid }
func (o OpCode) String() string {
	if name, ok := codeStrings[o]; ok {
		return name
	}
	return "unknown: " + strconv.FormatInt(int64(o), 10)
}
