package stream

// A Segment represents a slice of an information stream that represents the
// delta of information between the indicated positions.
// It is *not* implied that Segments sum linearly--if I attempt to append
// Segment(1-2) to Segment(2-3) inside a single byte buffer there is a good
// chance that data from the second delta should override data from the first.
// Only the source and receiver have enough contextual information to know how
// aggregation should work for a stream.
//
// It *is* implied that if segments have matching starting points that one can
// safely supercede the other. If Segment(1-2) and Segment(1-3) are both present
// it is safe to drop Segment(1-2) and report only the wider result. Sources
// can take advantage of this by using the Origin as a Segment.Start to clear
// any old messages and replace them with a summary, e.g. Segment(0-200) is a
// summary of all data up to position 200 that can be safely returned for any
// Position query inside that range.
type Segment struct {
	Start Position
	End   Position
}

// Empty signifies a segment with no content
var Empty = Segment{}

// EOF signifies a stream that has been closed
var EOF = Segment{Closed, Closed}

func (s Segment) Missing(pos Position) (Segment, bool) {
	if s.Start > pos {
		return Segment{pos, s.Start}, true
	}
	return Empty, false
}

// Distance returns the position count covered by the segment
func (s Segment) Distance() uint64 {
	if s.Start > s.End {
		return 0
	}
	return uint64(s.End - s.Start)
}
