package ticket

import (
	"io"
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
)

type sstats struct {
	createdOk    metrics.Count
	createdFail  metrics.Count
	redeemedOk   metrics.Count
	redeemedFail metrics.Count
	expired      metrics.Count
}

type Store interface {
	io.Closer
	Redeemer
	Tick()
	Request(stream.Credentials) (OpaqueBytes, error)
}

type store struct {
	stats  sstats
	age    time.Duration
	src    Factory
	recent map[string]*entry // 0 < age < tick
	old    map[string]*entry // tick < age < 2 * tick
	mutex  sync.Mutex
}

type entry struct {
	exp     time.Time
	creds   stream.Credentials
	claimed bool
}

var _ Store = (*store)(nil)

func NewStore(src Factory, age time.Duration, stats metrics.Tracker) Store {
	return &store{
		stats: sstats{
			createdOk:    stats.Count("tickets.created", []string{"result:ok"}),
			createdFail:  stats.Count("tickets.created", []string{"result:failed"}),
			redeemedOk:   stats.Count("tickets.redeemed", []string{"result:ok"}),
			redeemedFail: stats.Count("tickets.redeemed", []string{"result:failed"}),
			expired:      stats.Count("tickets.expired", []string{}),
		},
		age:    age,
		src:    src,
		recent: map[string]*entry{},
	}
}

func (s *store) Tick() {
	s.mutex.Lock()
	prev := s.old
	s.old = s.recent
	s.recent = map[string]*entry{}
	s.mutex.Unlock()
	if len(prev) > 0 {
		expired := int64(0)
		for _, t := range prev {
			if !t.claimed {
				expired++
			}
		}
		s.stats.expired.Add(expired)
	}
}

func (s *store) Close() error {
	s.Tick()
	s.Tick()
	return nil
}

func (s *store) Request(creds stream.Credentials) (OpaqueBytes, error) {
	key, err := s.src.Next()
	if err != nil {
		s.stats.createdFail.Add(1)
		return nil, err
	}
	exp := time.Now().Add(s.age)
	s.mutex.Lock()
	s.recent[string(key)] = &entry{exp: exp, creds: creds}
	s.mutex.Unlock()
	s.stats.createdOk.Add(1)
	return key, nil
}

// Redeem attempts to claim a ticket; it checks expiration and prior claims before
// it allows the claim to succeed
func (s *store) Redeem(method stream.AuthMethod, ticket stream.OpaqueBytes) stream.CredentialsPromise {
	creds := stream.NoPermissions()
	err := ErrInvalidTicket

	if method != stream.Reservation {
		s.stats.redeemedFail.Add(1)
		return stream.NewSyncCredentialsPromise(creds, protocol.ErrInvalidAuthMethod)
	}
	s.mutex.Lock()
	entry, ok := s.recent[string(ticket)]
	if !ok {
		entry, ok = s.old[string(ticket)]
	}
	if ok && !entry.claimed {
		entry.claimed = true
		err = nil
	}
	s.mutex.Unlock()
	if err == nil {
		if entry.exp.Before(time.Now()) {
			err = ErrInvalidTicket
		} else {
			creds = entry.creds
			err = s.src.Validate(ticket)
		}
	}
	if err == nil {
		s.stats.redeemedOk.Add(1)
	} else {
		s.stats.redeemedFail.Add(1)
	}
	return stream.NewSyncCredentialsPromise(creds, err)
}
