package ticket

import (
	"errors"
	"testing"
	"time"

	"code.justin.tv/devhub/e2ml/libs/metrics/devnull"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"github.com/stretchr/testify/assert"
)

type dummyFactory struct {
	ticket  OpaqueBytes
	nextErr error
	valid   error
}

var _ Factory = (*dummyFactory)(nil)

func (d *dummyFactory) Next() (OpaqueBytes, error) { return d.ticket, d.nextErr }
func (d *dummyFactory) Validate(OpaqueBytes) error { return d.valid }

func TestStore(t *testing.T) {
	readOnly := stream.NewCredentials("cid", stream.AddressScopes{stream.AnyAddress}, stream.AddressScopes{})

	t.Run("should allow redemption", func(t *testing.T) {
		f := &dummyFactory{OpaqueBytes("Admit One"), nil, nil}
		s := NewStore(f, time.Hour, devnull.NewTracker())

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		assert.Equal(t, f.ticket, ticket)

		data, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.NoError(t, err)
		assert.Equal(t, readOnly, data)

		assert.NoError(t, s.Close())
	})

	t.Run("should forward factory errors", func(t *testing.T) {
		expected := errors.New("expected")
		f := &dummyFactory{ticket: OpaqueBytes("Admit One"), nextErr: expected}
		s := NewStore(f, time.Hour, devnull.NewTracker())

		ticket, err := s.Request(readOnly)
		assert.Equal(t, expected, err)
		assert.Empty(t, ticket)

		creds, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.Equal(t, ErrInvalidTicket, err)
		assert.Equal(t, stream.NoPermissions(), creds)

		assert.NoError(t, s.Close())
	})

	t.Run("should report validation errors", func(t *testing.T) {
		expected := errors.New("expected")
		f := &dummyFactory{ticket: OpaqueBytes("Admit One"), valid: expected}
		s := NewStore(f, time.Hour, devnull.NewTracker())

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		assert.Equal(t, f.ticket, ticket)

		data, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.Equal(t, expected, err)
		assert.Equal(t, readOnly, data)

		assert.NoError(t, s.Close())
	})

	t.Run("should reject double claims", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Hour, devnull.NewTracker())

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		assert.Equal(t, f.ticket, ticket)

		creds, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.NoError(t, err)
		assert.Equal(t, readOnly, creds)

		creds, err = s.Redeem(stream.Reservation, ticket).Result()
		assert.Equal(t, stream.NoPermissions(), creds)
		assert.Equal(t, ErrInvalidTicket, err)

		assert.NoError(t, s.Close())
	})

	t.Run("should allow single tick entries", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Hour, devnull.NewTracker()).(*store)

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		s.Tick()

		data, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.NoError(t, err)
		assert.Equal(t, readOnly, data)

		assert.NoError(t, s.Close())
	})

	t.Run("should reject validation attempts", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Hour, devnull.NewTracker()).(*store)

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		s.Tick()

		creds, err := s.Redeem(stream.Validation, ticket).Result()
		assert.Equal(t, stream.NoPermissions(), creds)
		assert.Equal(t, protocol.ErrInvalidAuthMethod, err)

		assert.NoError(t, s.Close())
	})

	t.Run("should drop double tick entries", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Hour, devnull.NewTracker()).(*store)

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		s.Tick()
		s.Tick()

		creds, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.Equal(t, stream.NoPermissions(), creds)
		assert.Error(t, err)

		assert.NoError(t, s.Close())
	})

	t.Run("should rotate entries on a timer", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Millisecond, devnull.NewTracker())

		ticket, err := s.Request(readOnly)
		assert.NoError(t, err)
		time.Sleep(5 * time.Millisecond)

		creds, err := s.Redeem(stream.Reservation, ticket).Result()
		assert.Equal(t, stream.NoPermissions(), creds)
		assert.Error(t, err)

		assert.NoError(t, s.Close())
	})

	t.Run("should close safely", func(t *testing.T) {
		f := &dummyFactory{ticket: OpaqueBytes("Admit One")}
		s := NewStore(f, time.Millisecond, devnull.NewTracker())
		assert.NoError(t, s.Close())
		assert.NoError(t, s.Close())
	})
}
