package timeout

import (
	"math/rand"
	"sync/atomic"
	"time"
)

type precomputedSampler struct {
	samples []time.Duration
	counter int32
	length  int32
}

// pick a set of random samples in advance and loop over them; reduces overhead
// to maintain thread safety
func NewPrecomputedSampler(numSamples uint8, center, span time.Duration) Sampler {
	if span <= 0 || numSamples < 2 {
		return NewConstantSampler(center)
	}
	min := center - span/2
	if min < 0 {
		min = 0
	}
	delta := int64(span)

	p := &precomputedSampler{
		samples: make([]time.Duration, numSamples),
		length:  int32(numSamples),
	}
	for i := range p.samples {
		p.samples[i] = min + time.Duration(rand.Int63n(delta))
	}
	return p
}

func (p *precomputedSampler) Next() time.Duration {
	return p.samples[atomic.AddInt32(&p.counter, 1)%p.length]
}
