#!/bin/bash
set -e

show_usage_and_exit() {
  echo "Restart all tasks in the ECS service by updating the service with --force-new-deployment"
  echo "  Usage:   ./scripts/restart_service.sh <dev|prod> <greeter|pathfinder|threshold|source>"
  echo "  Example: ./scripts/restart_service.sh dev source"
  echo "Add --status to check when tasks have started, so it is easier to know when the restart is done"
  echo "  Example: ./scripts/restart_service.sh dev source --status"
  exit 1
}

export AWS_DEFAULT_REGION="us-west-2"
export AWS_PAGER="" # disable aws-cli v2 pagination with less

# param environment <dev|prod>
case "$1" in
  "dev")
    export AWS_PROFILE="twitch-eml-dev"
    export AWS_ACCOUNT="565915620853"
    case "$2" in
      "greeter")
        CLUSTER="DevGreeterService-ClusterEB0386A7-QKRWL0HD6EC1"
        SERVICE="DevGreeterService-FargateService7B4DE80D-OS2NPGSKPIU6"
        ;;
      "pathfinder")
        CLUSTER="DevPathfinderService-ClusterEB0386A7-MAPA266127HT"
        SERVICE="DevPathfinderService-FargateService7B4DE80D-J4DHKNQEYVGG"
        ;;
      "threshold")
        CLUSTER="DevThresholdService-ClusterEB0386A7-1WWWPEGJO3JU5"
        SERVICE="DevThresholdService-EC2ServiceF0CE72D0-169NHVHK8KUAF"
        ;;
      "source")
        CLUSTER="DevSourceService-ClusterEB0386A7-WS1S36AYOP2P"
        SERVICE="DevSourceService-EC2ServiceF0CE72D0-1BTN1PFDMW5EZ"
        ;;
    esac
    ;;
  "prod")
    export AWS_PROFILE="twitch-eml-prod"
    export AWS_ACCOUNT="342135511598"
    case "$2" in
      "greeter")
        CLUSTER="ProdGreeterService-ClusterEB0386A7-14F243CYT8VPN"
        SERVICE="ProdGreeterService-FargateService7B4DE80D-1GD0WR6GD96YC"
        ;;
      "pathfinder")
        CLUSTER="ProdPathfinderService-ClusterEB0386A7-1I254F53RW2S0"
        SERVICE="ProdPathfinderService-FargateService7B4DE80D-1X0Z9MHKGZWLY"
        ;;
      "threshold")
        CLUSTER="ProdThresholdService-ClusterEB0386A7-28PXMIVABFVD"
        SERVICE="ProdThresholdService-EC2ServiceF0CE72D0-NF0L7V6QN2OJ"
        ;;
      "source")
        CLUSTER="ProdSourceService-ClusterEB0386A7-M3EP3Q356332"
        SERVICE="ProdSourceService-EC2ServiceF0CE72D0-S09TNE632MCM"
        ;;
    esac
    ;;
esac
[[ -z $AWS_PROFILE ]] && { echo "ERROR: environment is invalid. Expected <dev|prod>"; show_usage_and_exit; }
[[ -z $CLUSTER ]] && { echo "ERROR: service is invalid. Expected <greeter|pathfinder|threshold|source>"; show_usage_and_exit; }

echo "Auth: Make sure to run mwinit from WPA2 or AmazonVPN, and then run (copy-paste) this ada command:"
echo "      ada credentials update --once --provider=isengard --role=admin --profile=$AWS_PROFILE --account=$AWS_ACCOUNT"
echo ""

if [[ -z $3 ]]; then
  echo ">> aws ecs update-service --cluster=$CLUSTER --service=$SERVICE --force-new-deployment"
  aws ecs update-service --cluster=$CLUSTER --service=$SERVICE --force-new-deployment

  echo ""
  echo "Restarting with a rolling re-deploy for all the tasks. It may take a while to complete."
  echo "To check restart status, re-run the script with $1 $2 --status"
fi

echo "Task Status:"
aws ecs describe-services --cluster=$CLUSTER --service=$SERVICE | jq '.services[0] | {desiredCount, runningCount, pendingCount}'

# Show startedAt for each task, so we can see if they have been started recently
echo "Tasks:"
TASK_ARNS=$(aws ecs list-tasks --cluster=$CLUSTER --service=$SERVICE | jq -r '[.taskArns[]]|join(" ")')
aws ecs describe-tasks --cluster $CLUSTER --tasks $TASK_ARNS | jq -r '.tasks[] | {taskArn, lastStatus, desiredStatus, startedAt}'

echo "To check again: $0 $1 $2 --status"
