#!/usr/bin/env bash
set -e

# Deploy E2ML doing the following steps:
# * Push a docker image to ECR. By default is the image tagged with the current git commit sha (as tagged by make build scripts).
# * Use the script https://github.com/silinternational/ecs-deploy to update the ECS Task Definition with the new docker image, and then updates the ECS Service to use the new version. ECS will add new tasks first, which requires extra capacity on Ec2 tasks (Fargate provides its own capacity). Then, when the new task is healthy, stops one of the old tasks.
# * Wait until all tasks are restarted and working with the new version.
#
# Requirements:
#  * Build docker image first (see build_docker.sh script)
#  * AWS CLI (http://docs.aws.amazon.com/cli/latest/userguide/tutorial-ec2-ubuntu.html#install-cli)
#  * jq: install with: brew install jq
#  * mwinit (and access to Amazon VPN): https://wiki.twitch.com/display/AD/Using+Isengard+from+the+command+line+or+programatically
#  * ada credentials: install with: toolbox install ada

show_usage_and_exit() {
  echo "Usage: $0 <env> <service> [git-tag]"
  echo "  <env>: dev|prod"
  echo "  <service>: greeter|pathfinder|threshold|source"
  echo "  [git-tag]: optional, is the current git log commit by default, used for docker image tag"
  echo "Example: $0 dev greeter"
  exit 1
}

ENV_NAME=$1
case $ENV_NAME in
  "dev")
    export AWS_PROFILE="twitch-eml-dev"
    export AWS_ACCOUNT="565915620853"
    ;;
  "prod")
    export AWS_PROFILE="twitch-eml-prod"
    export AWS_ACCOUNT="342135511598"
    ;;
  *)
    echo "invalid <env>: $1"
    show_usage_and_exit
    ;;
esac

SERVICE=$2
if [[ -z $SERVICE ]]; then
  echo "missing <service>"
  show_usage_and_exit
fi

GIT_TAG=$3
if [[ -z $GIT_TAG ]]; then
  GIT_TAG=`git log -1 --pretty=%H`
fi


echo "Login with ada. Make sure you are in Amazon VPN and ran mwinit before."
ada credentials update --once --provider=isengard --role=admin --profile=$AWS_PROFILE --account=$AWS_ACCOUNT

PROFILE="--profile $AWS_PROFILE"
REGION="--region us-west-2"
INPUT_IMAGE="messaging-proto/$SERVICE:$GIT_TAG"
echo "Docker image: $INPUT_IMAGE"

pushd "$(dirname $0)"

CLUSTER_ARN=$(aws resourcegroupstaggingapi get-resources ${PROFILE} ${REGION} --resource-type-filters ecs:cluster --tag-filters Key=service,Values=$SERVICE | jq -r ".ResourceTagMappingList[0].ResourceARN")
echo "Cluster ARN: $CLUSTER_ARN"
SERVICE_ARN=$(aws ecs list-services ${PROFILE} ${REGION} --cluster $CLUSTER_ARN | jq -r ".serviceArns[0]")
echo "Service ARN: $SERVICE_ARN"
REPOSITORY_ARN=$(aws resourcegroupstaggingapi get-resources ${PROFILE} ${REGION}  --resource-type-filters ecr:repository --tag-filters Key=service,Values=$SERVICE | jq -r ".ResourceTagMappingList[0].ResourceARN")
echo "Repository ARN: $REPOSITORY_ARN"

CLUSTER=$(aws ecs describe-clusters ${PROFILE} ${REGION} --cluster $CLUSTER_ARN | jq -r ".clusters[0].clusterName")
echo "Cluster: $CLUSTER"
SERVICE=$(aws ecs describe-services ${PROFILE} ${REGION} --cluster $CLUSTER_ARN --service $SERVICE_ARN | jq -r ".services[0].serviceName")
echo "Service: $SERVICE"
REPOSITORY=$(aws ecr describe-repositories ${PROFILE} ${REGION} | jq -r ".repositories[] | select ( .repositoryArn == \"$REPOSITORY_ARN\" ) | .repositoryUri")
echo "Repository: $REPOSITORY"

DEPLOY_IMAGE_LATEST="${REPOSITORY}:latest"
DEPLOY_IMAGE_TAGGED="${REPOSITORY}:${GIT_TAG}"

echo "ECR get-login ..."
aws ecr get-login-password ${REGION} | docker login --username AWS --password-stdin ${AWS_ACCOUNT}.dkr.ecr.us-west-2.amazonaws.com

# Deploy tagged version
echo "docker push $DEPLOY_IMAGE_TAGGED ..."
docker tag "$INPUT_IMAGE" "$DEPLOY_IMAGE_TAGGED"
docker push "$DEPLOY_IMAGE_TAGGED"

# Update services to use new version
echo "Update Services to use new version ..."
./ecs-deploy.sh ${PROFILE} ${REGION} --cluster "$CLUSTER" --service-name "$SERVICE"  --image "$DEPLOY_IMAGE_TAGGED" --timeout 60000

# After successful deploy, update the latest tag
if [[ "$DEPLOY_IMAGE_LATEST" != "$DEPLOY_IMAGE_TAGGED" ]]; then
  docker tag "$INPUT_IMAGE" "$DEPLOY_IMAGE_LATEST"
  docker push "$DEPLOY_IMAGE_LATEST"
fi

popd
