package main

import (
	"fmt"
	"log"
	"time"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/session/echo"
	"code.justin.tv/devhub/e2ml/libs/setup"
	"code.justin.tv/devhub/e2ml/libs/timeout"
	"code.justin.tv/devhub/e2ml/libs/websocket"
	"github.com/alexflint/go-arg"
)

type Config struct {
	Port        int           `arg:"env" default:"3004" help:"port for incoming connections"`
	HealthCheck *string       `arg:"--health-check,env:HEALTH_CHECK" default:"/health" help:"health check path"`
	Log         logging.Level `arg:"env" default:"info" help:"Logging level: [trace,debug,info,warning,error]"`
	CertFile    string        `arg:"--cert-file,env:CERT_FILE" help:"(optional) file containing TLS server cert"`
	KeyFile     string        `arg:"--key-file,env:KEY_FILE" help:"(optional) file containing TLS server key"`
}

func main() {
	var config Config
	arg.MustParse(&config)

	logger := logging.NewFilter(config.Log, log.Println).Log

	mgr := lifecycle.NewManager()
	defer func() { _ = mgr.ExecuteAll() }()
	lifecycle.SetDefaultPanicReporter(func(key interface{}, err error) { logger(logging.Error, err) })
	lifecycle.SetDefaultErrorReporter(func(key interface{}, err error) {
		logger(logging.Error, fmt.Sprintf("Shutdown error for %+v: %v", key, err))
	})

	service, err := websocket.NewServiceFactory(config.Port, &websocket.Settings{
		Certs:       setup.TLSConfig(config.CertFile, config.KeyFile, false),
		Lifecycle:   mgr,
		Logger:      logger,
		Timeout:     timeout.NewPrecomputedSampler(100, 10*time.Second, 1*time.Second),
		HealthCheck: config.HealthCheck,
	})(echo.Factory)
	if err == nil {
		err = service.Start()
	}
	if err != nil {
		setup.PanicWithMessage("Unable to start  service", err)
	}
	logger(logging.Info, "Stopping service on signal:", mgr.ListenForInterrupt())
	service.Stop()
	logger(logging.Info, "Waiting for connections to drop (max 5 seconds)")
	service.WaitForDrainingConnections(time.Now().Add(5 * time.Second))
}
