package main

import (
	"fmt"
	"time"

	"github.com/alexflint/go-arg"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/devhub/e2ml/libs/discovery/broker/balanced"
	"code.justin.tv/devhub/e2ml/libs/discovery/service"
	lhttp "code.justin.tv/devhub/e2ml/libs/http"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/peering"
	"code.justin.tv/devhub/e2ml/libs/session"
	"code.justin.tv/devhub/e2ml/libs/setup"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/timeout"
	"code.justin.tv/devhub/e2ml/libs/websocket"
	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
)

const (
	defaultHealthCheck = "/health"
	s2sAudience        = "eml-greeter"
)

type Config struct {
	metrics.GoArgSource
	EnvName string `arg:"--env-name,env:ENV_NAME" help:"Current environment: [local,dev,prod]"`

	CertFile      string        `arg:"--cert-file,env:CERT_FILE" help:"(optional) file containing TLS server cert"`
	KeyFile       string        `arg:"--key-file,env:KEY_FILE" help:"(optional) file containing TLS server key"`
	Insecure      bool          `help:"FOR LOCAL USE ONLY: allow connections to untrusted servers"`
	Log           logging.Level `arg:"env" default:"info" help:"Logging level: [trace,debug,info,warning,error]"`
	MetricsHost   string        `arg:"--metrics-host,env:METRICS_HOST" default:"localhost" help:"hostname for metric reporting"`
	MetricsMethod string        `arg:"--metrics-method,env:METRICS_METHOD" default:"none" help:"method for metric collection [none,logged,twitchtelemetry]"`
	MetricsPeriod time.Duration `arg:"--metrics-period,env:METRICS_PERIOD" default:"10s" help:"period between metrics reporting ticks"`

	ClientAuthMethods string             `arg:"--client-auth-methods,env:CLIENT_AUTH_METHODS" default:"validator" help:"comma delimited method list [empty,fake,fake-validator,validator]"`
	ClientPort        int                `arg:"--client-port,env:CLIENT_PORT" default:"8000" help:"port for client connections"`
	ClientS2sSecret   stream.OpaqueBytes `arg:"--client-s2s-secret,env:CLIENT_S2S_SECRET" help:"secret for client s2s auth"`
	ValidatorURL      setup.URL          `arg:"--validator-url,env:VALIDATOR_URL" default:"https://validator" help:"Url for client validator auth"`

	HostAuthMethod      string             `arg:"--host-auth-method,env:HOST_AUTH_METHOD" default:"s2s" help:"authorization method for host connections [s2s,fake]"`
	HostElectionTimeout time.Duration      `arg:"--host-election-timeout:env:HOST_ELECTION_TIMEOUT" default:"20s" help:"timeout for stream source elections"`
	HostPort            int                `arg:"--host-port,env:HOST_PORT" default:"3000" help:"port for host connections"`
	HostS2sSecret       stream.OpaqueBytes `arg:"--host-s2s-secret,env:HOST_S2S_SECRET" help:"secret for host s2s auth"`

	PeerAuthMethod string             `arg:"--peer-auth-method,env:PEER_AUTH_METHOD" default:"s2s" help:"authorization method for peer connections [s2s,fake]"`
	PeerData       string             `arg:"--peer-discovery-data,env:PEER_DISCOVERY_DATA" help:"method dependent data string to locate peers"`
	PeerLocalName  string             `arg:"--peer-local-name,env:PEER_LOCAL_NAME" help:"name of this instance in peer list, used to exclude from connection attempts"`
	PeerMethod     string             `arg:"--peer-discovery-method,env:PEER_DISCOVERY_METHOD" default:"hardcoded" help:"method for peer discovery"`
	PeerPort       int                `arg:"--peer-port,env:PEER_PORT" default:"3010" help:"port for peer connections"`
	PeerS2sSecret  stream.OpaqueBytes `arg:"--peer-s2s-secret,env:PEER_S2S_SECRET" help:"secret for peer s2s auth"`
}

func main() {
	var config Config
	arg.MustParse(&config)

	logger := setup.Logger(config.Log)

	mgr := lifecycle.NewManager()
	defer func() { _ = mgr.ExecuteAll() }()
	lifecycle.SetDefaultPanicReporter(func(key interface{}, err error) { logger(logging.Error, err) })
	lifecycle.SetDefaultErrorReporter(func(key interface{}, err error) {
		logger(logging.Error, fmt.Sprintf("Shutdown error for %+v: %v", key, err))
	})

	tPid := identifier.ProcessIdentifier{
		Machine:  config.MetricsTaskARN,
		Service:  "E2MLGreeter",
		Stage:    config.EnvName,
		Substage: "primary",
		Region:   "us-west-2",
	}

	metrics := setup.MetricsTracker(config.MetricsMethod, tPid, logger)
	mgr.RegisterHook(metrics, func() error {
		err := metrics.Close()
		time.Sleep(time.Second) // allow metrics to drain
		return err
	})
	mgr.TickUntilClosed(metrics.Tick, config.MetricsPeriod)
	metrics.Count("lifecycle", []string{"action:startup"}).Add(1)

	gostats := setup.GoStatsCollector(tPid, 30*time.Second, logger)
	gostats.Start()
	mgr.RegisterHook(gostats, func() error {
		gostats.Stop()
		return nil
	})

	wstls := setup.TLSConfig(config.CertFile, config.KeyFile, config.Insecure)

	// shut down peer list source last ...
	serverList, nameResolver := setup.ServerList(config.PeerMethod, config.PeerLocalName, config.PeerData, config.PeerPort, logger)
	mgr.RegisterHook(serverList, serverList.Close)

	// after open peer connections ...
	peerService := websocket.NewServiceFactory(config.PeerPort, &websocket.Settings{
		Certs:     wstls,
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewPrecomputedSampler(100, 10*time.Second, 1*time.Second),
	})

	peerClients := websocket.NewClientResolver(&websocket.Settings{
		Certs:     wstls,
		Lifecycle: mgr,
		Logger:    logger,
		Timeout:   timeout.NewPrecomputedSampler(100, 10*time.Second, 1*time.Second),
	})

	// after the peer manager ...
	peerFactory := func(logic session.BindingFactory) peering.Manager {
		peers := peering.NewManager(logic, peerClients, peerService, serverList, nameResolver, logger)
		mgr.RegisterHook(peers, peers.Shutdown)
		return peers
	}

	// after closing elections ...
	validatorURL := setup.Unwrap(config.ValidatorURL)
	balancedBroker, err := balanced.NewBroker(
		peerFactory,
		setup.AuthSource("peer", config.PeerAuthMethod, s2sAudience, s2sAudience, config.PeerS2sSecret, logger),
		setup.AuthReader("peer", config.PeerAuthMethod, s2sAudience, validatorURL, config.PeerS2sSecret, metrics, logger),
		setup.AuthReader("host", config.HostAuthMethod, s2sAudience, validatorURL, config.HostS2sSecret, metrics, logger),
		setup.AuthReader("client", config.ClientAuthMethods, s2sAudience, validatorURL, config.ClientS2sSecret, metrics, logger),
		100*time.Millisecond,
		config.HostElectionTimeout,
		metrics,
		logger,
	)
	if err != nil {
		setup.PanicWithMessage("Unable to start the broker", err)
	}
	mgr.RegisterHook(balancedBroker, balancedBroker.Close)

	// after reporter connections last ...
	healthCheck := defaultHealthCheck
	reporters, err := websocket.NewServiceFactory(config.HostPort, &websocket.Settings{
		Certs:       wstls,
		Lifecycle:   mgr,
		Logger:      logger,
		Timeout:     timeout.NewPrecomputedSampler(100, 10*time.Second, 1*time.Second),
		HealthCheck: &healthCheck,
	})(balancedBroker.Factory())
	if err == nil {
		err = reporters.Start()
	}
	if err != nil {
		setup.PanicWithMessage("Unable to start reporter service", err)
	}

	// after the connection broker clients (note: this service may be removed)
	clients, err := lhttp.NewServiceFactory(config.ClientPort, &lhttp.Settings{
		Certs:     wstls,
		Lifecycle: mgr,
		Logger:    logger,
	})(service.BuildServer(setup.AuthExtractor(config.ClientAuthMethods), balancedBroker))
	if err == nil {
		err = clients.Start()
	}
	if err != nil {
		setup.PanicWithMessage("ListenAndServe error:", err)
	}

	logger(logging.Info, "Exiting on signal:", mgr.ListenForInterrupt())
	metrics.Count("lifecycle", []string{"action:drain"}).Add(1)
	clients.WaitForDrainingConnections(time.Now().Add(10 * time.Second))
	metrics.Count("lifecycle", []string{"action:shutdown"}).Add(1)
}
