package source

import (
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
)

type historyMap map[stream.AddressKey]*historyEntry
type historyFactory func(stream.Address) (stream.History, error)
type addressCallback func(stream.Address)
type entryCallback func(*historyEntry)

type hstats struct {
	histories metrics.Aggregator
	tracker   metrics.Tracker
}

// this struct tracks and releases histories that no longer have any
// attached audience after a preset time by rotating history maps
type historyManager struct {
	stats     hstats
	factory   historyFactory
	onDestroy addressCallback
	reporter  discovery.HostReporter
	tick      time.Duration
	active    historyMap
	recent    historyMap
	old       historyMap
	close     chan struct{}
	mutex     sync.Mutex
}

func newHistoryManager(reporter discovery.HostReporter, tick time.Duration, factory historyFactory, onDestroy addressCallback, tracker metrics.Tracker) *historyManager {
	return &historyManager{
		stats: hstats{
			histories: tracker.Aggregator("source.histories", []string{}),
			tracker:   tracker,
		},
		reporter:  reporter,
		tick:      tick,
		factory:   factory,
		onDestroy: onDestroy,
		active:    make(historyMap),
		recent:    make(historyMap),
		old:       make(historyMap),
	}
}

func (h *historyManager) GetOrCreate(addr stream.Address) (stream.History, error) {
	return h.createOrUpdate(addr, false, nil)
}

func (h *historyManager) OnJoined(addr stream.Address) {
	h.createOrUpdate(addr, true, func(t *historyEntry) { t.OnJoined() })
}

func (h *historyManager) OnParted(addr stream.Address) {
	key := addr.Key()
	h.mutex.Lock()
	defer h.mutex.Unlock()
	if entry, ok := h.active[key]; ok && !entry.OnParted() {
		delete(h.active, key)
		h.recent[key] = entry
	}
}

func (h *historyManager) OnRequested(addr stream.Address) {
	h.createOrUpdate(addr, true, func(t *historyEntry) { t.OnRequested() })
}

func (h *historyManager) OnReleased(addr stream.Address) {
	key := addr.Key()
	h.mutex.Lock()
	defer h.mutex.Unlock()
	if entry, ok := h.active[key]; ok && !entry.OnReleased() {
		delete(h.active, key)
		h.recent[key] = entry
	}
}

func (h *historyManager) limitMetric(addr stream.Address) metrics.Count {
	return h.stats.tracker.Count("source.rate_limited", []string{})
}

func (h *historyManager) createOrUpdate(addr stream.Address, activate bool, action entryCallback) (history stream.History, err error) {
	key := addr.Key()
	h.mutex.Lock()
	entry, found := h.findAndRefresh(key, activate)
	if found {
		history = entry.History()
		if action != nil {
			action(entry)
		}
	} else {
		history, err = h.factory(addr)
		if err == nil {
			entry = newHistoryEntry(history)
			if action != nil {
				action(entry)
			}
			if activate {
				h.active[key] = entry
			} else {
				h.recent[key] = entry
			}
		}
	}
	h.mutex.Unlock()
	if !found {
		h.stats.histories.Add(1)
		src, _ := entry.inner.Last()
		h.reporter.AddSupported(stream.AddressSourceMap{addr.Key(): src})
	}
	return
}

func (h *historyManager) revoke(addr stream.Address) error {
	key := addr.Key()
	h.mutex.Lock()
	entry, found := h.active[key]
	delete(h.active, key)
	if !found {
		entry, found = h.recent[key]
		delete(h.recent, key)
		if !found {
			entry, found = h.old[key]
			delete(h.old, key)
		}
	}
	h.mutex.Unlock()
	if found {
		src, _ := entry.inner.Last()
		h.reporter.DropSupported(stream.AddressSourceMap{key: src})
		h.onDestroy(addr)
		h.stats.histories.Add(-1)
	}
	return nil
}

// must be called from within a lock
func (h *historyManager) findAndRefresh(key stream.AddressKey, activate bool) (*historyEntry, bool) {
	if entry, found := h.active[key]; found {
		return entry, true
	}
	if entry, found := h.recent[key]; found {
		if activate {
			delete(h.recent, key)
			h.active[key] = entry
		}
		return entry, true
	}
	entry, found := h.old[key]
	if !found {
		return nil, false
	}
	delete(h.old, key)
	// always refresh if it was old
	if activate {
		h.active[key] = entry
	} else {
		h.recent[key] = entry
	}
	return entry, true
}

func (h *historyManager) Start() {
	h.mutex.Lock()
	defer h.mutex.Unlock()
	if h.close == nil {
		h.close = make(chan struct{})
		go h.run()
	}
}

func (h *historyManager) Stop() {
	h.mutex.Lock()
	defer h.mutex.Unlock()
	ch := h.close
	h.close = nil
	close(ch)
}

func (h *historyManager) run() {
	h.mutex.Lock()
	closer := h.close
	tick := h.tick
	h.mutex.Unlock()
	if closer == nil {
		return
	}
	ticker := time.NewTicker(tick)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			h.onTick()
		case <-closer:
			return
		}
	}
}

func (h *historyManager) onTick() {
	h.mutex.Lock()
	dead := h.old
	h.old = h.recent
	h.recent = make(historyMap)
	h.mutex.Unlock()
	length := len(dead)
	if length == 0 {
		return
	}
	scopes := make(stream.AddressSourceMap)
	for _, wrapper := range dead {
		addr := wrapper.History().Address()
		src, _ := wrapper.History().Last()
		scopes[addr.Key()] = src
		h.onDestroy(addr)
	}
	h.reporter.DropSupported(scopes)
	h.stats.histories.Add(-int64(length))
}
