package source

import (
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/discovery"
	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/audience"
	"code.justin.tv/devhub/e2ml/libs/stream/history"
	"code.justin.tv/devhub/e2ml/libs/stream/protocol"
	"code.justin.tv/devhub/e2ml/libs/stream/scheduler"
	"code.justin.tv/devhub/e2ml/libs/ticket"
)

type sourceIDGenerator func() stream.SourceID

type logic struct {
	newSourceID sourceIDGenerator
	aud         stream.Audience
	trackers    map[stream.AddressKey]*stream.Tracker
	histories   *historyManager
	limitCount  uint8
	limitPeriod time.Duration
	mutex       sync.Mutex
}

var _ discovery.SourceLogic = (*logic)(nil)

func NewServer(idsGen sourceIDGenerator, tickets ticket.Redeemer, reporter discovery.HostReporter, limitCount uint8, limitPeriod, historyExpiration, warnBeforeAuthExpiration time.Duration, metrics metrics.Tracker, log logging.Function) audience.Remote {
	logicFactory := newServerLogic(idsGen, reporter, limitCount, limitPeriod, historyExpiration, metrics)
	return audience.NewRemote(logicFactory, scheduler.NewInline(), tickets, warnBeforeAuthExpiration, metrics, log)
}

func newServerLogic(idGenerator sourceIDGenerator, reporter discovery.HostReporter, limitCount uint8, limitPeriod, historyExpiration time.Duration, metrics metrics.Tracker) func(stream.Audience) stream.ServerLogic {
	return func(aud stream.Audience) stream.ServerLogic {
		logic := &logic{
			newSourceID: idGenerator,
			aud:         aud,
			limitCount:  limitCount,
			limitPeriod: limitPeriod,
			trackers:    make(map[stream.AddressKey]*stream.Tracker),
		}
		logic.histories = newHistoryManager(reporter, historyExpiration, logic.createHistory, logic.destroyTracker, metrics)
		reporter.SetSourceLogic(logic)
		logic.histories.Start()
		return logic
	}
}

func (l *logic) LoadFactor() uint64 { return l.aud.LoadFactor() }

func (l *logic) Shutdown() error {
	l.histories.Stop()
	return nil
}

func (l *logic) ServeAddress(addr stream.Address) (stream.AddressScopes, error) {
	l.GetHistory(addr) // make sure a history exists
	return stream.AddressScopes{addr}, nil
}

func (l *logic) RevokeAddress(addr stream.Address) (stream.AddressScopes, error) {
	return nil, l.histories.revoke(addr)
}

func (l *logic) OnAddressJoined(addr stream.Address) error {
	l.histories.OnJoined(addr)
	return nil
}

func (l *logic) OnAddressParted(addr stream.Address) error {
	l.histories.OnParted(addr)
	return nil
}

func (l *logic) OnAddressRequested(addr stream.Address) error {
	l.histories.OnRequested(addr)
	return nil
}

func (l *logic) OnAddressReleased(addr stream.Address) error {
	l.histories.OnReleased(addr)
	return nil
}

func (l *logic) OnSendRequested(addr stream.Address, data []byte, isDelta bool, receipt stream.MutableTrackerPromise) {
	err := error(protocol.ErrServiceUnavailable)
	src, pos := l.getOrCreateTracker(addr).Next()
	// mark complete if send panics
	defer setReceipt(receipt, src, pos, &err)
	var topic stream.Topic
	topic, err = l.aud.ForAddress(addr)
	if err == nil {
		err = topic.Send(src, pos.ToSegment(isDelta), data)
	}
}

func (l *logic) GetHistory(addr stream.Address) (stream.History, error) {
	// reuses existing history if applicable
	return l.histories.GetOrCreate(addr)
}

func setReceipt(receipt stream.MutableTrackerPromise, src stream.SourceID, pos stream.Position, err *error) {
	receipt.Set(stream.CreateTracker(src, pos), *err)
}

func (l *logic) createHistory(addr stream.Address) (stream.History, error) {
	var h stream.History
	if l.limitCount > 0 && l.limitPeriod > 0 {
		h = history.NewRateLimited(addr, l.limitCount, l.limitPeriod, l.histories.limitMetric(addr))
	} else {
		h = history.NewSingleEntry(addr)
	}
	// allow source to service this address
	l.aud.Enable(stream.AddressScopes{addr})
	// initialize source ID for this channel
	h.Write(stream.NewMessage(addr, l.newSourceID(), stream.Empty, nil))
	return h, nil
}

func (l *logic) getOrCreateTracker(addr stream.Address) *stream.Tracker {
	key := addr.Key()
	l.mutex.Lock()
	tracker, existed := l.trackers[key]
	if !existed {
		tracker = l.createTracker(addr)
		l.trackers[key] = tracker
	}
	l.mutex.Unlock()
	return tracker
}

func (l *logic) destroyTracker(addr stream.Address) {
	l.mutex.Lock()
	delete(l.trackers, addr.Key())
	l.mutex.Unlock()
	l.aud.Disable(stream.AddressScopes{addr})
}

func (l *logic) createTracker(addr stream.Address) *stream.Tracker {
	tracker := new(stream.Tracker)
	if history, err := l.histories.GetOrCreate(addr); err == nil {
		src, last := history.Last()
		// starting tracker position with an offset from origin enables automatic reporting
		// of gaps if this source is picking up delta traffic that originated from a different
		// source
		if last == stream.Origin {
			last += 1
		}
		tracker.Set(src, last)
	}
	return tracker
}
