package threshold

import (
	"sync"
	"time"

	"code.justin.tv/devhub/e2ml/libs/logging"
	"code.justin.tv/devhub/e2ml/libs/metrics"
	"code.justin.tv/devhub/e2ml/libs/stream"
	"code.justin.tv/devhub/e2ml/libs/stream/audience"
	"code.justin.tv/devhub/e2ml/libs/stream/history"
	"code.justin.tv/devhub/e2ml/libs/stream/scheduler"
	"code.justin.tv/devhub/e2ml/libs/ticket"
)

// TODO : service side mathmatics for gauges
type stats struct {
	fanoutCount        metrics.Count
	fanoutTotalElapsed metrics.Count
}

type logic struct {
	stats   stats
	aud     stream.Audience
	reg     stream.Registry
	logger  logging.Function
	writers map[stream.AddressKey]stream.Writer
	mutex   sync.Mutex
}

var _ stream.Listener = (*logic)(nil)

func NewServer(reg stream.Registry, tickets ticket.Redeemer, warnBeforeAuthExpiration time.Duration, metrics metrics.Tracker, logger logging.Function) audience.Remote {
	logic := newServerLogic(reg, metrics, logger)
	return audience.NewRemote(logic, scheduler.NewInline(), tickets, warnBeforeAuthExpiration, metrics, logger)
}

func newServerLogic(reg stream.Registry, metrics metrics.Tracker, logger logging.Function) stream.ServerLogicFactory {
	return func(aud stream.Audience) stream.ServerLogic {
		aud.Enable(stream.AddressScopes{stream.AnyAddress}) // prime to accept all addreses
		return &logic{
			stats: stats{
				fanoutCount:        metrics.Count("threshold.fanout.count", []string{}),
				fanoutTotalElapsed: metrics.Count("threshold.fanout.elapsed", []string{}),
			},
			aud:     aud,
			reg:     reg,
			writers: make(map[stream.AddressKey]stream.Writer),
			logger:  logger,
		}
	}
}

func (l *logic) LoadFactor() uint64 { return l.aud.LoadFactor() }

func (l *logic) Shutdown() error {
	return nil
}

func (l *logic) OnAddressJoined(addr stream.Address) error {
	_, err := l.reg.Reader(addr).Join(l).Result()
	return err
}

func (l *logic) OnAddressParted(addr stream.Address) error {
	_, err := l.reg.Reader(addr).Leave(l).Result()
	return err
}

func (l *logic) OnAddressRequested(addr stream.Address) error {
	key := addr.Key()
	l.mutex.Lock()
	if _, ok := l.writers[key]; !ok {
		l.writers[key] = l.reg.Writer(addr)
	}
	l.mutex.Unlock()
	return nil
}

func (l *logic) OnAddressReleased(addr stream.Address) error {
	key := addr.Key()
	l.mutex.Lock()
	writer := l.writers[key]
	delete(l.writers, key)
	l.mutex.Unlock()
	if writer != nil {
		return writer.Close()
	}
	return nil
}

func (l *logic) OnSendRequested(addr stream.Address, data []byte, isDelta bool, receipt stream.MutableTrackerPromise) {
	key := addr.Key()
	l.mutex.Lock()
	writer := l.writers[key]
	l.mutex.Unlock()
	// TODO : better scheduler model with appropriate backpressure
	if writer != nil {
		go func() {
			sentPromise := writer.Send(data, isDelta)
			tacker, err := sentPromise.Result() // block until sent
			receipt.Set(tacker, err)
		}()
	} else {
		l.logger(logging.Debug, "Attempt to use unregistered writer")
		receipt.Set(stream.Untracked, stream.ErrWriterAlreadyClosed)
	}
}

func (l *logic) GetHistory(addr stream.Address) (stream.History, error) {
	return history.NewSingleEntry(addr), nil
}

func (l *logic) Current(addr stream.Address) (stream.SourceID, stream.Position) {
	if t, err := l.aud.ForAddress(addr); err == nil {
		return t.Current()
	}
	return stream.None, stream.Origin
}

func (l *logic) OnDataLost(desc stream.MessageDescription) bool {
	// TODO : proper forwarded notification; in practice we should have a Send
	// immediately afterward
	return true
}

func (l *logic) OnDataReceived(msg stream.Message) bool {
	start := time.Now()

	topic, err := l.aud.ForAddress(msg.Address())
	if err != nil {
		l.logger(logging.Error, "threshold.logic.OnDataReceived.Audience.ForAddress: ", err)
		return false
	}

	err = topic.Forward(msg)
	if err != nil {
		l.logger(logging.Error, "threshold.logic.OnDataReceived.Topic.Forward: ", err)
		return false
	}

	l.stats.fanoutTotalElapsed.Add(int64(time.Since(start)))
	l.stats.fanoutCount.Add(1)
	return true
}

func (l *logic) OnStreamClosed(addr stream.Address, err error) bool {
	t, ferr := l.aud.ForAddress(addr)
	if ferr != nil {
		return false
	}
	t.Close(err)
	return true
}
