// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

import { DynamodbStack } from './stacks/dynamodb';
import { EcrRepositoryStack } from './stacks/ecr-repository';
import { ECSClusterStack } from './stacks/ecs-cluster';
import { ECSEC2ServiceStack } from './stacks/ecs-ec2-service';
import { ECSFargateServiceStack } from './stacks/ecs-fargate-service';
import { S3BucketStack } from './stacks/s3-bucket';
import { VpcStack } from './stacks/vpc';

interface EnvProps {
  prefix: string;
  env: cdk.Environment;
  ec2AsgInstanceType: ec2.InstanceType;
  ec2AsgCapacity: number;
  vpcId: string;
  vpcPrivSubnets: ec2.SubnetAttributes[];
  redisElasticacheEndpoint: string;
}

const app = new cdk.App();

createStacks(app, {
  prefix: 'Dev',
  env: {
    account: '947140302170', // twitch-devhub-testing
    region: 'us-west-2',
  },
  ec2AsgInstanceType: new ec2.InstanceType('m5.large'),
  ec2AsgCapacity: 0, // used for dev02 test instance
  vpcId: 'vpc-043141871adcd673d',
  vpcPrivSubnets: [
    { subnetId: 'subnet-0fdd43ec64b209ac7', availabilityZone: 'us-west-2a', routeTableId: 'rtb-0eea41b29309bb55b' }, // 10.0.96.0/19
    { subnetId: 'subnet-00aa0035c7b33c837', availabilityZone: 'us-west-2b', routeTableId: 'rtb-06407afc10682fc3c' }, // 10.0.128.0/19
    { subnetId: 'subnet-04d6334d08485ec6f', availabilityZone: 'us-west-2c', routeTableId: 'rtb-0b24bd7f81e60a972' }, // 10.0.160.0/19
  ],
  redisElasticacheEndpoint: 'e2topics-0001-001.5s8cjl.0001.usw2.cache.amazonaws.com',
});

// createStacks for each environment
function createStacks(scope: cdk.App, props: EnvProps) {
  const { prefix, env } = props;

  const vpcStack = new VpcStack(scope, prefix + 'Vpc', {
    env,
    vpcId: props.vpcId,
    vpcPrivSubnets: props.vpcPrivSubnets,
  });

  const ecrRepoStack = new EcrRepositoryStack(app, prefix + 'E2TopicsRepo', { env, repositoryName: 'e2topics' }); // repo name as used by scripts/docker_push.sh

  const s3BucketStack = new S3BucketStack(app, prefix + 'E2TopicsS3Bucket', { env, bucketName: "e2topics" });

  const dynamodbStack = new DynamodbStack(app, prefix + 'E2TopicsS3DynamoDB', { env, tableName: "e2topics" });

  const clusterStack = new ECSClusterStack(app, prefix + 'E2TopicsECSCluster', {
    env,
    vpc: vpcStack.vpc,
    ec2Asg: {
      instanceType: props.ec2AsgInstanceType,
      minCapacity: props.ec2AsgCapacity,
      maxCapacity: props.ec2AsgCapacity,
    },
  });

  new ECSFargateServiceStack(app, prefix + 'E2TopicsServiceDev01', {
    env,
    testInstance: 'dev01',
    vpc: vpcStack.vpc,
    cluster: clusterStack.cluster,
    ecrRepo: ecrRepoStack.ecrRepo,
    s3Bucket: s3BucketStack.bucket,
    redisElasticacheEndpoint: props.redisElasticacheEndpoint,
    dynamodbTable: dynamodbStack.table,
  });

  new ECSEC2ServiceStack(app, prefix + 'E2TopicsServiceDev02', {
    env,
    testInstance: 'dev02',
    vpc: vpcStack.vpc,
    cluster: clusterStack.cluster,
    ecrRepo: ecrRepoStack.ecrRepo,
    s3Bucket: s3BucketStack.bucket,
    redisElasticacheEndpoint: props.redisElasticacheEndpoint,
    dynamodbTable: dynamodbStack.table,
  });
}

