import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';

import { E2TopicsContainerDefinition } from './ecs-fargate-service';

interface ECSServiceStackProps extends cdk.StackProps {
  testInstance: string; // "dev01", "dev02", etc., identifies the Fargate Service, and DynamicConf file in S3

  vpc: ec2.IVpc;
  cluster: ecs.ICluster;
  ecrRepo: ecr.IRepository;
  s3Bucket: s3.IBucket; // to read dynamic config file
  redisElasticacheEndpoint: string;
  dynamodbTable: dynamodb.ITable;
}

export class ECSEC2ServiceStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ECSServiceStackProps) {
    super(scope, id, props);

    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDefinition', {
      networkMode: ecs.NetworkMode.AWS_VPC,
    });

    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    props.s3Bucket.grantRead(taskDefinition.taskRole); // allow to read dynamic config file form the S3 bucket
    props.dynamodbTable.grantReadWriteData(taskDefinition.taskRole); // allow to use DynamoDB table

    const container = taskDefinition.addContainer('TaskContainer', E2TopicsContainerDefinition(props));
    container.addUlimits({
      name: ecs.UlimitName.NOFILE, // number of open files, needed to handle socket connections
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Service
    new ecs.Ec2Service(this, 'TaskService', {
      serviceName: `e2topics_${props.testInstance}`,
      cluster: props.cluster, // cluster must have an ASG with enough capacity
      taskDefinition,
      assignPublicIp: false,
      desiredCount: 0, // updated through /scripts/run.sh
      placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
    });
  }
}
