import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';

interface ECSServiceStackProps extends cdk.StackProps {
    testInstance: string; // "dev01", "dev02", etc., identifies the Fargate Service, and DynamicConf file in S3

    vpc: ec2.IVpc;
    cluster: ecs.ICluster;
    ecrRepo: ecr.IRepository;
    s3Bucket: s3.IBucket; // to read dynamic config file
    redisElasticacheEndpoint: string;
    dynamodbTable: dynamodb.ITable;
}

export class ECSFargateServiceStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ECSServiceStackProps) {
    super(scope, id, props);

    const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDefinition', {
      cpu: 4096, // 4 vCPU
      memoryLimitMiB: 8192, // 8 GB (minimum available for the cpu value)
    });

    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    props.s3Bucket.grantRead(taskDefinition.taskRole); // allow to read dynamic config file form the S3 bucket
    props.dynamodbTable.grantReadWriteData(taskDefinition.taskRole); // allow to use DynamoDB table

    const container = taskDefinition.addContainer('TaskContainer', E2TopicsContainerDefinition(props));
    container.addUlimits({
      name: ecs.UlimitName.NOFILE, // number of open files, needed to handle socket connections
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Service
    new ecs.FargateService(this, 'TaskService', {
      serviceName: `e2topics_${props.testInstance}`,
      cluster: props.cluster,
      taskDefinition,
      assignPublicIp: false,
      desiredCount: 0, // updated through /scripts/run.sh
    });
  }
}

export function E2TopicsContainerDefinition(props: ECSServiceStackProps): ecs.ContainerDefinitionOptions {
    return {
        image: ecs.ContainerImage.fromEcrRepository(props.ecrRepo, 'latest'), // Requires props.ecrRepo to exist and have a registered docker image
        logging: new ecs.AwsLogDriver({ streamPrefix: 'e2topics' }),
        memoryReservationMiB: 1024, // 1 GB
        environment: {
          TARGETENV: "dev",
          TESTINSTANCE: props.testInstance,
          DYNAMICCONFFILE: `dynamic_config/${props.testInstance}.toml`, // e.g. "dynamic_config/dev01.toml",
          DYNAMICCONFS3BUCKET: props.s3Bucket.bucketName,
          REDISCLUSTERADDRS: 'e2topics.5s8cjl.clustercfg.usw2.cache.amazonaws.com:6379',
          DYNAMODBTABLE: props.dynamodbTable.tableName,
        }
    };
}
