package config

import (
	"io"
	"io/ioutil"
	"os"
	"reflect"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	toml "github.com/pelletier/go-toml"
)

const (
	checkUpdatesInterval = 5 * time.Second
)

type DynamicConfig struct {
	TaskCount  int    // number of tasks running with this test instance configuration
	PubsubType string // implementation used by publishers and subscribers: "STREAMS", "GETSET", "DYNAMODB", ...

	Channels             int // total number of channels to publish messages (one publisher per channel)
	ChannelOffset        int // first channel number on this test instance. Can be used to split write traffic on different test instances.
	AvgViewersPerChannel int // viewers per channel, exponentially distributed on the channels (more viewers on lower channels)

	MsgsPerMinute      int // messages per minute that are published on each channel. Set to zero to make this test subscribers-only
	MsgLen             int // length of messages publised (this is roughly the size in Bytes)
	NewConnsPerSeccond int // how many new channels are added/removed per seccon when the number is ramped up/down

	TaskNumberKey    string // redis key used to uniquely identify each taskNumber
	TaskNumberExpire int    // seconds to expire and restart the taskNumber key

	LogLevel string // trace|debug|info|warn|error, default "info"
}

type DynamicConfigMngr struct {
	s3Bucket string
	filePath string

	s3Cli *s3.S3

	loadNextAt time.Time
	loading    bool
	LoadedLast *DynamicConfig

	loadResultVal *DynamicConfig
	loadResultErr error
	loadResultMux sync.Mutex
}

func MustNewDynamicConfigMngr(s3Bucket string, filePath string) *DynamicConfigMngr {
	m := &DynamicConfigMngr{
		s3Bucket: s3Bucket,
		filePath: filePath,
	}
	if s3Bucket != "" {
		sess, err := session.NewSession(&aws.Config{Region: aws.String("us-west-2")}) // reads credentials from environment variables or .aws/credentials file
		if err != nil {
			panic(err)
		}
		m.s3Cli = s3.New(sess)
	}
	return m
}

// TickAndCheckUpdatesAsync should be called from the main loop at quick intervals with the current time (now).
// Returns the new loaded configuration if there are changes since the last time it was loaded.
func (m *DynamicConfigMngr) TickAndCheckUpdatesAsync(now time.Time) (*DynamicConfig, error) {
	if now.Before(m.loadNextAt) {
		return nil, nil // last load was too recent
	}

	if !m.loading {
		m.loading = true
		go func() { // load async
			loaded, loadErr := m.Load()
			m.SetLoadResult(loaded, loadErr)
		}()
		return nil, nil // loading ...
	}

	loaded, loadErr := m.GetLoadResult()
	if loaded == nil && loadErr == nil {
		return nil, nil // still loading ...
	}

	m.SetLoadResult(nil, nil)
	m.loading = false
	m.loadNextAt = now.Add(checkUpdatesInterval) // new interval to check for updates after loaded (the time to load was added to the real interval)

	if loadErr != nil {
		return nil, loadErr
	}
	if reflect.DeepEqual(loaded, m.LoadedLast) { // shallow equality check
		return nil, nil // nothing changed since last check
	}
	m.LoadedLast = loaded
	return loaded, nil
}

func (m *DynamicConfigMngr) SetLoadResult(dc *DynamicConfig, err error) {
	m.loadResultMux.Lock()
	defer m.loadResultMux.Unlock()
	m.loadResultVal, m.loadResultErr = dc, err
}

func (m *DynamicConfigMngr) GetLoadResult() (*DynamicConfig, error) {
	m.loadResultMux.Lock()
	defer m.loadResultMux.Unlock()
	return m.loadResultVal, m.loadResultErr
}

func (m *DynamicConfigMngr) MustLoad() *DynamicConfig {
	conf, err := m.Load()
	if err != nil {
		panic(err)
	}
	return conf
}

func (m *DynamicConfigMngr) Load() (*DynamicConfig, error) {
	if m.s3Bucket != "" {
		return m.LoadFromS3()
	} else {
		return m.LoadFromFile()
	}
}

func (m *DynamicConfigMngr) LoadFromS3() (*DynamicConfig, error) {
	result, err := m.s3Cli.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(m.s3Bucket),
		Key:    aws.String(m.filePath),
	})
	if err != nil {
		return nil, err
	}
	defer result.Body.Close()
	return ParseToml(result.Body)
}

func (m *DynamicConfigMngr) LoadFromFile() (*DynamicConfig, error) {
	file, err := os.Open(m.filePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()
	return ParseToml(file)
}

func ParseToml(body io.Reader) (*DynamicConfig, error) {
	bodyBytes, err := ioutil.ReadAll(body)
	if err != nil {
		return nil, err
	}
	dc := &DynamicConfig{}
	err = toml.Unmarshal(bodyBytes, dc)
	return dc, err
}
