package config

import (
	"strings"

	"github.com/alexflint/go-arg"
)

// ParseGoArgConfig reads program args and environment variables
func ParseGoArgConfig() GoArgConfig {
	conf := GoArgConfig{}
	arg.MustParse(&conf)
	return conf
}

type GoArgConfig struct {
	TestInstance string `arg:"env:TESTINSTANCE" default:"local" help:"Name to identify the fargate service and DynamicConf (should match). E.g. dev01, dev02, etc."`

	DynamicConfFile     string `arg:"env:DYNAMICCONFFILE,-f" default:"config/dynamic_config/local.toml" help:"Path for the TOML file with dynamic configuration. Used as object key on the S3 bucket, or as local file path if S3 bucket is not specified."`
	DynamicConfS3Bucket string `arg:"env:DYNAMICCONFS3BUCKET,-b" help:"S3 bucket name for the remote TOML file with dynamic configuration. If not specified, the file is loaded from a local file path."`

	RedisClusterAddrs string `arg:"env:REDISCLUSTERADDRS,-a" default:":30001,:30002,:30003" help:"comma separated list of Redis Cluster addresses"`
	DynamoDBTable     string `arg:"env:DYNAMODBTABLE,-d" default:"e2topics" help:"DynamoDB table when using the DYNAMODB PubsubType implementation from DynamicConf"`
}

func (conf GoArgConfig) IsLocal() bool {
	return conf.TestInstance == "local"
}

func (conf GoArgConfig) RedisClusterAddrsList() []string {
	return strings.Split(conf.RedisClusterAddrs, ",")
}
