package e2topics

import (
	"context"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"

	"github.com/graph-gophers/dataloader"
)

type DynamoBatchLoader struct {
	DynamodbTbl string
	DynamodbCli *dynamodb.DynamoDB

	loader *dataloader.Loader
	writer *dataloader.Loader
}

func NewDynamoBatchLoader(dynamodbTbl string, dynamodbCli *dynamodb.DynamoDB) *DynamoBatchLoader {
	d := &DynamoBatchLoader{
		DynamodbTbl: dynamodbTbl,
		DynamodbCli: dynamodbCli,
	}
	d.loader = dataloader.NewBatchedLoader(
		d.BatchLoaderFn,
		dataloader.WithInputCapacity(100),        // Max Dynamodb batch size for BatchGetItem
		dataloader.WithBatchCapacity(100),        // Max Dynamodb batch size for BatchGetItem
		dataloader.WithWait(16*time.Millisecond), // amount of time to wait before triggering a batch
		dataloader.WithClearCacheOnBatch(),
		dataloader.WithCache(&dataloader.NoCache{}),
	)
	d.writer = dataloader.NewBatchedLoader(
		d.BatchWriterFn,
		dataloader.WithInputCapacity(25),         // Max Dynamodb batch size for BatchWriteItem
		dataloader.WithBatchCapacity(25),         // Max Dynamodb batch size for BatchWriteItem
		dataloader.WithWait(10*time.Millisecond), // amount of time to wait before triggering a batch
		dataloader.WithClearCacheOnBatch(),
		dataloader.WithCache(&dataloader.NoCache{}),
	)
	return d
}

// GetItem blocks until a full back is loaded and then returns the item for that key.
func (d *DynamoBatchLoader) GetItem(key string) (map[string]*dynamodb.AttributeValue, error) {
	thunk := d.loader.Load(context.Background(), dataloader.StringKey(key))
	result, err := thunk() // block until loaded
	if item, ok := result.(map[string]*dynamodb.AttributeValue); ok {
		return item, err
	}
	return nil, err
}

func (d *DynamoBatchLoader) PutItem(item map[string]*dynamodb.AttributeValue) error {
	thunk := d.writer.Load(context.Background(), &itemAsDataloaderKey{item})
	_, err := thunk() // block until all batch is written
	return err
}

func (d *DynamoBatchLoader) BatchLoaderFn(ctx context.Context, keys dataloader.Keys) []*dataloader.Result {
	// Batch load from DynamoDB
	dynamoResp, err := d.DynamodbCli.BatchGetItem(&dynamodb.BatchGetItemInput{
		RequestItems: map[string]*dynamodb.KeysAndAttributes{
			d.DynamodbTbl: {
				Keys: ToDynamodBatchGetItems(keys),
			},
		},
	})
	if err != nil {
		return dataloaderResultsWithNilData(keys, err)
	}

	itemsMap := map[string]map[string]*dynamodb.AttributeValue{}
	for _, item := range dynamoResp.Responses[d.DynamodbTbl] {
		if key := item["id"].S; key != nil {
			itemsMap[*key] = item
		}
	}

	// if we need to track unprocessed keys, we can read them here
	// unprocessedKeys, ok := dynamoResp.UnprocessedKeys[d.DynamodbTbl]

	var results []*dataloader.Result
	for _, key := range keys { // make sure the results length is the same as keys length
		data := itemsMap[key.String()] // map[string]*dynamodb.AttributeValue
		var err error
		if data == nil {
			err = fmt.Errorf("DynamoDB: unprocessed key")
		}
		results = append(results, &dataloader.Result{
			Data:  data,
			Error: err,
		})
	}
	return results
}

func ToDynamodBatchGetItems(keys dataloader.Keys) []map[string]*dynamodb.AttributeValue {
	ddbKeys := make([]map[string]*dynamodb.AttributeValue, len(keys))
	for i, key := range keys {
		ddbKeys[i] = map[string]*dynamodb.AttributeValue{
			"id": {S: aws.String(key.String())},
		}
	}
	return ddbKeys
}

func (d *DynamoBatchLoader) BatchWriterFn(ctx context.Context, items dataloader.Keys) []*dataloader.Result {
	// Batch Write to DynamoDB
	_, err := d.DynamodbCli.BatchWriteItem(&dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			d.DynamodbTbl: ToDynamodBatchPutItems(items),
		},
	})
	return dataloaderResultsWithNilData(items, err)
}

func ToDynamodBatchPutItems(items dataloader.Keys) []*dynamodb.WriteRequest {
	ddbRequests := make([]*dynamodb.WriteRequest, len(items))
	for i, dataloaderKey := range items {
		item := dataloaderKey.Raw().(map[string]*dynamodb.AttributeValue)

		ddbRequests[i] = &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{Item: item},
		}
	}
	return ddbRequests
}

func dataloaderResultsWithNilData(keys dataloader.Keys, err error) []*dataloader.Result {
	var results []*dataloader.Result
	for range keys {
		results = append(results, &dataloader.Result{
			Data:  nil,
			Error: err,
		})
	}
	return results
}

// Implements dataloader.Key so it can be passed to the BatchWriterFn
type itemAsDataloaderKey struct {
	item map[string]*dynamodb.AttributeValue
}

func (i *itemAsDataloaderKey) String() string {
	if key := i.item["id"].S; key != nil {
		return *key
	}
	return ""
}

func (i *itemAsDataloaderKey) Raw() interface{} {
	return i.item
}
