package e2topics

import (
	"encoding/binary"
	"strconv"
	"time"

	"github.com/mediocregopher/radix/v3"
	"github.com/rs/zerolog/log"
)

type Message struct {
	Bytes  []byte
	SentAt time.Time
}

type MessageAsDynamodbItem struct {
	ID  string `dynamodbav:"id"`
	Msg []byte `dynamodbav:"msg"`
	Ts  string `dynamodbav:"ts"`
}

func NewRandomMessage(length int, t time.Time) *Message {
	return &Message{
		Bytes:  NewRandomMsgOfSize(length),
		SentAt: t,
	}
}

func (m *Message) EncodeAsBytes() []byte {
	const lenInt64 = 8
	bytes := make([]byte, lenInt64+len(m.Bytes))
	binary.BigEndian.PutUint64(bytes[0:], uint64(TimeToUnixMilli(m.SentAt)))
	copy(bytes[lenInt64:], m.Bytes)
	return bytes
}

func (m *Message) DecodeFromBytes(bytes []byte) {
	const lenInt64 = 8
	if len(bytes) < lenInt64 {
		m.SentAt = time.Time{}
		m.Bytes = []byte{}
		return
	}
	sentAtMillis := binary.BigEndian.Uint64(bytes[0:])
	m.SentAt = TimeFromUnixMilli(int64(sentAtMillis))
	m.Bytes = bytes[lenInt64:]
}

func (m *Message) SentAtStr() string {
	return strconv.FormatInt(TimeToUnixMilli(m.SentAt), 10)
}

func SentAtTimeFromStr(sentAtStr string) time.Time {
	sentAtInt64, err := strconv.ParseInt(sentAtStr, 10, 64)
	if err != nil {
		return time.Time{}
	} else {
		return TimeFromUnixMilli(sentAtInt64)
	}
}

func DecodeFromStreamEntry(entry *radix.StreamEntry) *Message {
	if entry == nil {
		return nil
	}

	msg := &Message{}
	if val, ok := entry.Fields["Value"]; ok {
		msg.Bytes = []byte(val)
	}
	if sentAt, ok := entry.Fields["SentAtUnixMillis"]; ok {
		msg.SentAt = SentAtTimeFromStr(sentAt)
		if msg.SentAt.IsZero() {
			log.Error().Msgf("Could not parse msg: SentAtUnixMillis: %v", sentAt)
		}
	} else {
		log.Warn().Msgf("DecodeFromStreamEntry: SentAtUnixMillis key not found in entry.Fields: %v", entry.Fields)
	}
	return msg
}

// helpers

const charset = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"

func NewRandomMsgOfSize(length int) []byte {
	b := make([]byte, length)
	for i := range b {
		b[i] = charset[i%len(charset)]
		// b[i] = charset[rand.Intn(len(charset))] // warning: using rand creates a global lock bottleneck
	}
	return b
}

func TimeToUnixMilli(t time.Time) int64 {
	ns := t.UnixNano()
	if ns < 0 {
		return (ns - 999999) / 1000000
	}
	return ns / 1000000
}

func TimeFromUnixMilli(ms int64) time.Time {
	return time.Unix(0, 1000000*ms)
}
