package e2topics

import (
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/mediocregopher/radix/v3"
)

type Publisher struct {
	RedisCli        radix.Client
	RedisBatch      *RedisBatchLoader
	DynamodbCli     *dynamodb.DynamoDB
	DynamodbTbl     string
	DynamoBatch     *DynamoBatchLoader
	PubsubType      string // "STREAMS", "GETSET", ...
	Channel         string
	MsgLen          int
	PublishInterval time.Duration
	OnMsgSent       func(p *Publisher, msg *Message, d time.Duration, err error)

	mutex sync.Mutex
}

func (p *Publisher) StartPublishing() {
	go func() {
		for {
			start := time.Now()
			msg := NewRandomMessage(p.GetMsgLen(), start)
			err := p.SendMsg(context.Background(), msg)
			dur := time.Since(start)
			if err != nil {
				p.OnMsgSent(p, nil, dur, fmt.Errorf("Publisher.SendMsg: %w", err))
				time.Sleep(5 * time.Second) // wait a little extra to allow recovering form service errors
			} else {
				p.OnMsgSent(p, msg, dur, nil)
			}

			time.Sleep(p.GetPublishInterval() - dur)
		}
	}()
}

func (p *Publisher) GetPublishInterval() time.Duration {
	p.mutex.Lock()
	defer p.mutex.Unlock()
	return p.PublishInterval
}

func (p *Publisher) SetPublishInterval(d time.Duration) {
	p.mutex.Lock()
	defer p.mutex.Unlock()
	p.PublishInterval = d
}

func (p *Publisher) GetMsgLen() int {
	p.mutex.Lock()
	defer p.mutex.Unlock()
	return p.MsgLen
}

func (p *Publisher) SetMsgLen(n int) {
	p.mutex.Lock()
	defer p.mutex.Unlock()
	p.MsgLen = n
}

// SendMsg sends a message to Redis Stream.
func (p *Publisher) SendMsg(ctx context.Context, msg *Message) error {
	if p.PubsubType == "STREAMS" {
		return p.SendMsgStreams(ctx, msg)
	} else if p.PubsubType == "GETSET" {
		return p.SendMsgGetSet(msg)
	} else if p.PubsubType == "GETSETPIPE" || p.PubsubType == "GETSETLUA" {
		return p.SendMsgGetSetPipe(msg)
	} else if p.PubsubType == "GETSETBATCH" {
		return p.SendMsgGetSetBatch(msg)
	} else if p.PubsubType == "DYNAMODB" {
		return p.SendMsgDynamodb(msg)
	} else if p.PubsubType == "DYNAMODBBATCH" {
		return p.SendMsgDynamodbBatch(msg)
	} else {
		return fmt.Errorf("Unknown PubsubType: %q", p.PubsubType)
	}
}

func (p *Publisher) SendMsgStreams(ctx context.Context, msg *Message) error {
	key := p.PubsubType + "-" + p.Channel

	return p.RedisCli.Do(radix.FlatCmd(nil, "XADD", key, "MAXLEN", "~", "1000", "*",
		"Value", msg.Bytes,
		"SentAtUnixMillis", msg.SentAtStr()))
}

func (p *Publisher) SendMsgGetSet(msg *Message) error {
	key := p.PubsubType + "-" + p.Channel
	const expireSecs = 10

	return p.RedisCli.Do(radix.FlatCmd(nil, "SET", key, msg.EncodeAsBytes(), "EX", expireSecs))
}

func (p *Publisher) SendMsgGetSetPipe(msg *Message) error {
	key := p.PubsubType + "-" + p.Channel
	keyTime := fmt.Sprintf("{%s}.time", key)
	keyIncr := fmt.Sprintf("{%s}.incr", key)
	const expireSecs = 10

	// SET msg on key with EXPIRE
	// INCR +1 on keyIncr with EXPIRE
	// The keyIncr can be checked by subscribers when polling to make sure the key value is new.
	return p.RedisCli.Do(radix.Pipeline(
		radix.FlatCmd(nil, "SET", key, msg.Bytes, "EX", expireSecs),
		radix.FlatCmd(nil, "SET", keyTime, msg.SentAtStr(), "EX", expireSecs),
		radix.FlatCmd(nil, "INCR", keyIncr),
		radix.FlatCmd(nil, "EXPIRE", keyIncr, expireSecs),
	))
}

func (p *Publisher) SendMsgGetSetBatch(msg *Message) error {
	key := p.PubsubType + "-" + p.Channel
	return p.RedisBatch.Set(key, msg.EncodeAsBytes())
}

func (p *Publisher) SendMsgDynamodb(msg *Message) error {
	key := p.PubsubType + "-" + p.Channel
	msgItem := MessageAsDynamodbItem{
		ID:  key,
		Msg: msg.Bytes,
		Ts:  msg.SentAtStr(),
	}
	item, err := dynamodbattribute.MarshalMap(msgItem)
	if err != nil {
		return err
	}

	_, err = p.DynamodbCli.PutItem(&dynamodb.PutItemInput{
		TableName: aws.String(p.DynamodbTbl),
		Item:      item,
	})
	return err
}

func (p *Publisher) SendMsgDynamodbBatch(msg *Message) error {
	key := p.PubsubType + "-" + p.Channel
	msgItem := MessageAsDynamodbItem{
		ID:  key,
		Msg: msg.Bytes,
		Ts:  msg.SentAtStr(),
	}
	item, err := dynamodbattribute.MarshalMap(msgItem)
	if err != nil {
		return err
	}

	return p.DynamoBatch.PutItem(item)
}
