package e2topics

import (
	"sync"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
	poller "code.justin.tv/amzn/TwitchTelemetryPollingCollector"
	"code.justin.tv/devhub/e2topics/config"
)

type Statter interface {
	// Inc reports "Count"
	Inc(metric string, n int)

	// Duration reports "Seconds"
	Duration(metric string, d time.Duration)

	// Gauge reports a "Count" but only when a time interval has passed (e.g. 10 seconds),
	// otherwise the value is ignored. Gauges are meant to be aggregated with "Average".
	Gauge(metric string, n int)
}

func NewTelemetryStatter(conf config.GoArgConfig) *TelemetryStatter {
	tPid := identifier.ProcessIdentifier{
		Machine:  conf.TestInstance,
		Service:  "E2Topics",
		Stage:    "dev",
		Substage: "primary",
		Region:   "us-west-2",
	}
	sender := cw.NewUnbuffered(&tPid, nil)
	sampleBuilder := telemetry.SampleBuilder{ProcessIdentifier: tPid}
	sampleObserver := telemetry.NewBufferedAggregator(20*time.Second, 100000, time.Minute, sender, nil)

	// Start collecting Go system stats
	goPoller := poller.NewGoStatsPollingCollector(10*time.Second, &sampleBuilder, sampleObserver, nil)
	goPoller.Start()

	return &TelemetryStatter{
		r: &telemetry.SampleReporter{
			SampleBuilder:  sampleBuilder,
			SampleObserver: sampleObserver,
		},
		gaugePeriod:        10 * time.Second,
		gaugeNextReportsAt: map[string]time.Time{},
	}
}

type TelemetryStatter struct {
	r                  *telemetry.SampleReporter
	gaugePeriod        time.Duration
	gaugeNextReportsAt map[string]time.Time
	gaugeMutext        sync.RWMutex
}

func (s *TelemetryStatter) Inc(metric string, n int) {
	s.r.Report(metric, float64(n), telemetry.UnitCount)
}
func (s *TelemetryStatter) Duration(metric string, d time.Duration) {
	s.r.ReportDurationSample(metric, d)
}
func (s *TelemetryStatter) Gauge(metric string, n int) {
	var nextReportAt time.Time
	s.gaugeMutext.RLock()
	nextReportAt = s.gaugeNextReportsAt[metric]
	s.gaugeMutext.RUnlock()

	now := time.Now()
	if now.After(nextReportAt) {
		s.gaugeMutext.Lock()
		s.gaugeNextReportsAt[metric] = now.Add(s.gaugePeriod)
		s.gaugeMutext.Unlock()

		s.r.Report(metric, float64(n), telemetry.UnitCount)
	}
}

type NullStatter struct{}

func (s *NullStatter) Inc(metric string, n int)                {}
func (s *NullStatter) Duration(metric string, d time.Duration) {}
func (s *NullStatter) Gauge(metric string, n int)              {}
