package e2topics

import (
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/mediocregopher/radix/v3"
	"github.com/rs/zerolog/log"
)

const (
	xReadBlockTimeout  = 2 * time.Second  // how long to wait for new messages.
	xReadRetryWait     = 5 * time.Second  // wait to try again, either on error or after block-timeout.
	getsetPollInterval = 1 * time.Second  // check if the message has been updated.
	getsetRetryWait    = 10 * time.Second // wait a little longer if the stream has not started or there's an error.
)

type Subscriber struct {
	RedisCli    radix.Client
	RedisBatch  *RedisBatchLoader
	DynamodbCli *dynamodb.DynamoDB
	DynamoBatch *DynamoBatchLoader
	DynamodbTbl string
	PubsubType  string // "STREAMS", "GETSET", ...
	Channel     string
	OnMsgRead   func(s *Subscriber, msg *Message, d time.Duration, err error)
}

func (s *Subscriber) StartReading() {
	if s.PubsubType == "STREAMS" {
		go s.StartReadingStreams()
	} else if s.PubsubType == "GETSET" {
		go s.StartReadingGetSet()
	} else if s.PubsubType == "GETSETLUA" {
		go s.StartReadingGetSetLua()
	} else if s.PubsubType == "GETSETPIPE" {
		go s.StartReadingGetSetPipe()
	} else if s.PubsubType == "GETSETBATCH" {
		go s.StartReadingGetSetBatch()
	} else if s.PubsubType == "DYNAMODB" {
		go s.StartReadingDynamodb()
	} else if s.PubsubType == "DYNAMODBBATCH" {
		go s.StartReadingDynamodbBatch()
	} else {
		log.Error().Msgf("Subscriber StartReading: invaild PubsubType: %s", s.PubsubType)
	}
}

func (s *Subscriber) NewStreamReader(lastEntryID *radix.StreamEntryID) radix.StreamReader {
	key := s.PubsubType + "-" + s.Channel
	return radix.NewStreamReader(s.RedisCli, radix.StreamReaderOpts{
		Streams: map[string]*radix.StreamEntryID{key: lastEntryID}, // use nil to start from "$", only read new messages
		Block:   xReadBlockTimeout,
		Count:   100, // we only care about the last value, but we need to fast forward in case there are a lot of writes in between reads
	})
}

func (s *Subscriber) StartReadingStreams() {
	// NOTE: check first if there's already a last value in the stream with
	// XREVRANGE key + - 1
	// But we can skip that to simplify because when I tested with it it was insignificant when compared to the XREAD blocks

	var lastEntryID *radix.StreamEntryID = nil // star reading from "$", only new entries
	streamReader := s.NewStreamReader(lastEntryID)

	for {
		start := time.Now()
		_, entries, ok := streamReader.Next()
		if !ok {
			s.OnMsgRead(s, nil, time.Since(start), streamReader.Err())

			streamReader = s.NewStreamReader(lastEntryID)
			time.Sleep(xReadRetryWait)
			continue
		}

		if entry := findMostRecent(entries); entry != nil {
			lastEntryID = &entry.ID
			s.OnMsgRead(s, DecodeFromStreamEntry(entry), time.Since(start), nil)
		} else {
			time.Sleep(xReadRetryWait) // nothing to read so far (block timeout), the game is probably not publishing yet, so no rush to re-connect
		}
	}
}

func (s *Subscriber) StartReadingGetSet() {
	key := s.PubsubType + "-" + s.Channel
	for {
		start := time.Now()
		var msgBytes []byte
		err := s.RedisCli.Do(radix.Cmd(&msgBytes, "GET", key))
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		msg := &Message{}
		msg.DecodeFromBytes(msgBytes)

		s.OnMsgRead(s, msg, time.Since(start), nil)
		time.Sleep(getsetPollInterval)
	}
}

func (s *Subscriber) StartReadingGetSetLua() {
	key := s.PubsubType + "-" + s.Channel
	keyTime := fmt.Sprintf("{%s}.time", key)
	keyIncr := fmt.Sprintf("{%s}.incr", key)

	// Check keyIncr first to see if the value has been updated since last check,
	// if the value was incremented, then return the full msg from the key.
	// The script returns a list of strings with [incrVal, sentAt, msg]
	getIfNew := radix.NewEvalScript(3, `
		local incrVal = redis.call("GET", KEYS[1])
		if incrVal == false then -- stream is not active
			return {"-1", "0", ""}
		end

		if incrVal == ARGV[1] then -- value exists but was not updated
			return {incrVal, "0", ""}
		end

		local sentAt = redis.call("GET", KEYS[2])
		local msgStr = redis.call("GET", KEYS[3])
		return {incrVal, sentAt, msgStr}
	`)

	incrVal := "0"
	for {
		start := time.Now()
		var result []string
		err := s.RedisCli.Do(getIfNew.Cmd(&result, keyIncr, keyTime, keyIncr, incrVal))
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		newIncrVal := result[0]
		sentAtMillisStr := result[1]
		msgStr := result[2]

		if newIncrVal == "-1" {
			// Stream is not active, wait a little longer to try again
			time.Sleep(getsetRetryWait)
			continue
		}
		if newIncrVal != incrVal { // a new msg was published
			incrVal = newIncrVal

			s.OnMsgRead(s, &Message{
				Bytes:  []byte(msgStr),
				SentAt: SentAtTimeFromStr(sentAtMillisStr),
			}, time.Since(start), nil)
		}

		time.Sleep(getsetPollInterval)
	}
}

func (s *Subscriber) StartReadingGetSetPipe() {
	key := s.PubsubType + "-" + s.Channel
	keyTime := fmt.Sprintf("{%s}.time", key)
	keyIncr := fmt.Sprintf("{%s}.incr", key)

	incrVal := int64(0) // used to check when the msg is updated
	for {
		start := time.Now()
		var msgBytes []byte
		var msgTimeStr string
		var newIncrVal int64
		commands := radix.Pipeline(
			radix.Cmd(&msgBytes, "GET", key),
			radix.Cmd(&msgTimeStr, "GET", keyTime),
			radix.Cmd(&newIncrVal, "GET", keyIncr),
		)
		err := s.RedisCli.Do(commands)
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		if newIncrVal != incrVal { // a new msg was published
			incrVal = newIncrVal
			msgTime := SentAtTimeFromStr(msgTimeStr)
			s.OnMsgRead(s, &Message{Bytes: msgBytes, SentAt: msgTime}, time.Since(start), nil)
		}

		time.Sleep(getsetPollInterval)
	}
}

func (s *Subscriber) StartReadingGetSetBatch() {
	key := s.PubsubType + "-" + s.Channel
	for {
		start := time.Now()
		msgBytes, err := s.RedisBatch.Get(key)
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		msg := &Message{}
		msg.DecodeFromBytes(msgBytes)

		s.OnMsgRead(s, msg, time.Since(start), nil)
		time.Sleep(getsetPollInterval)
	}
}

func (s *Subscriber) StartReadingDynamodb() {
	key := s.PubsubType + "-" + s.Channel

	for {
		start := time.Now()
		result, err := s.DynamodbCli.GetItem(&dynamodb.GetItemInput{
			TableName: aws.String(s.DynamodbTbl),
			Key: map[string]*dynamodb.AttributeValue{
				"id": {S: aws.String(key)},
			},
		})
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		msg, err := UnmarshalDynamodbItem(result.Item)
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}
		if msg == nil { // expired
			time.Sleep(getsetRetryWait) // just retry later: wait until publisher starts sending messages
			continue
		}
		s.OnMsgRead(s, msg, time.Since(start), nil)
		time.Sleep(getsetPollInterval)
	}
}

func (s *Subscriber) StartReadingDynamodbBatch() {
	key := s.PubsubType + "-" + s.Channel
	for {
		start := time.Now()
		item, err := s.DynamoBatch.GetItem(key)
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}

		msg, err := UnmarshalDynamodbItem(item)
		if err != nil {
			s.OnMsgRead(s, nil, time.Since(start), err)
			time.Sleep(getsetRetryWait)
			continue
		}
		if msg == nil { // expired
			time.Sleep(getsetRetryWait) // just retry later: wait until publisher starts sending messages
			continue
		}
		s.OnMsgRead(s, msg, time.Since(start), nil)
		time.Sleep(getsetPollInterval)
	}
}

func UnmarshalDynamodbItem(item map[string]*dynamodb.AttributeValue) (*Message, error) {
	msgItem := MessageAsDynamodbItem{}
	err := dynamodbattribute.UnmarshalMap(item, &msgItem)
	if err != nil {
		return nil, err
	}

	sentAt := SentAtTimeFromStr(msgItem.Ts)
	if time.Since(sentAt) > 5*time.Minute {
		return nil, nil
	}

	return &Message{
		Bytes:  msgItem.Msg,
		SentAt: sentAt,
	}, nil
}

func findMostRecent(entries []radix.StreamEntry) *radix.StreamEntry {
	var recent *radix.StreamEntry
	for _, e := range entries {
		if recent == nil || recent.ID.Before(e.ID) {
			recent = &e
		}
	}
	return recent
}
