package main

import (
	"fmt"

	"github.com/mediocregopher/radix/v3"
)

// Number of keys to find.
// Must be a divisor of 16384: [8192, 4096, 2048, 1024, 512, 256, 128, 64, 32, 16, 8, 4, 2, 1].
// Each key will map into an equally distributed Hash Slot, and can be used later
// to group Redis keys, ensuring they are handled in the same shard. Useful for MGET, XADD, etc.
const keysToFind = 128

const (
	redisHashSlots = 16384                       // all available redis hash slots
	keySlotRange   = redisHashSlots / keysToFind // each key will handle this range of possible slots

	charset = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"
)

func main() {
	fmt.Printf("Find %d keys\n", keysToFind)
	fmt.Printf("where each key maps to a Redis Cluster hash slot (CRC16 MOD %d) that equally distributed for each %d slots in the same area.\n", redisHashSlots, keySlotRange)
	foundKeys := make([]string, keysToFind)

	const maxKeyLength = 5 // hard stop if we don't find all the keys
	for k := 1; k <= maxKeyLength; k++ {
		fmt.Printf("Checking keys of length %d ...\n", k)
		numFound := 0

		forEachCombinationOf(charset, k, func(key string) bool {
			redisSlot := crc16Slot(key)
			slot := redisSlot / keySlotRange

			if foundKeys[slot] == "" { // new empty slot found
				foundKeys[slot] = string(key)
			}
			numFound = numFoundKeys(foundKeys)
			return numFound == keysToFind // if all keys were found, stop searching
		})

		fmt.Printf("%d/%d key slots found\n", numFound, keysToFind)
		if numFound == keysToFind {
			break
		}
	}

	fmt.Println("Result as sample code:")
	fmt.Println("")
	fmt.Println("const redisHashSlots =", redisHashSlots, "// all available Redis Cluster Hash slots")
	fmt.Println("")
	fmt.Println("// keyGroupTags are key tags that map into a known Redis Cluster slot")
	fmt.Println("var keyGroupTags = []string{")
	for slot := 0; slot < keysToFind; slot++ {
		key := foundKeys[slot]
		redisSlot := crc16Slot(key)
		lowerBound := slot * keySlotRange
		upperBound := ((slot + 1) * keySlotRange) - 1
		fmt.Printf("    %q, \t// groups [%d:%d] on hash slot %d\n", key, lowerBound, upperBound, redisSlot)
	}
	fmt.Println("}")
	fmt.Println("")
	fmt.Println("// keyGroupIdx returns the index such as keyGroupTags[keyGroupIdx] is the keyGroupTag for this key")
	fmt.Println("func keyGroupIdx(key []byte) int {")
	fmt.Println("	hashSlot := int(radix.CRC16(key)) % redisHashSlots // same Cluster HASH SLOT that Redis would calculate.")
	fmt.Println("   keyGroupSize := redisHashSlots / len(keyGroupTags) // each keyGroupTag handles a group of slots")
	fmt.Println("   return hashSlot / keyGroupSize")
	fmt.Println("}")
}

func numFoundKeys(foundKeys []string) int {
	n := 0
	for _, key := range foundKeys {
		if key != "" {
			n += 1
		}
	}
	return n
}

func crc16Slot(key string) int {
	return int(radix.CRC16([]byte(key))) % redisHashSlots
}

func forEachCombinationOf(str string, k int, callback func(string) bool) {
	_ = forEachCombinationOf_(str, "", len(str), k, callback)
}

// print all possible strings of length k
func forEachCombinationOf_(str string, prefix string, n int, k int, callback func(string) bool) bool {
	if k == 0 {
		return callback(prefix) // Base case: k = 0
	}

	// One by one add all characters from charset, and recursively call for k = k-1
	for i := 0; i < n; i++ {
		newPrefix := prefix + string(str[i]) // Next character added

		// k is decreased, because we have added a new character
		completed := forEachCombinationOf_(str, newPrefix, n, k-1, callback)
		if completed {
			return completed
		}
	}
	return false
}
