#!/bin/bash
set -e

export AWS_DEFAULT_REGION="us-west-2"
export AWS_PROFILE="twitch-devhub-testing"
export AWS_ACCOUNT="947140302170"
export ECS_CLUSTER="DevE2TopicsECSCluster-ClusterEB0386A7-SNMENyH5M64P"
export ECS_SERVICE_PREFIX="e2topics"
export S3_BUCKET="e2topics"

exit_with_usage() {
    echo "Run, stop or update dynamic_conf for ECS services. The script does the following:"
    echo " 1. Uploads dynamic config to S3"
    echo " 2. Updates the TaskCount of the ECS service"
    echo "Examples:"
    echo "  ./scripts/run.sh dev01     # run number of tasks from the config file's TaskCount property"
    echo "  ./scripts/run.sh dev01 5   # run 5 instances of the service"
    echo "  ./scripts/run.sh dev01 0   # stop the service (run zero tasks)"
    exit 1
}

TESTNAME="$1"
[[ -z "$TESTNAME" ]] && exit_with_usage
LOCAL_FILE="./config/dynamic_config/$TESTNAME.toml"
S3RMT_FILE="s3://$S3_BUCKET/dynamic_config/$TESTNAME.toml"
ECS_SERVICE="${ECS_SERVICE_PREFIX}_${TESTNAME}"

TASK_COUNT="$2"
if [[ -z "$TASK_COUNT" ]]; then
    TASK_COUNT=$(cat $LOCAL_FILE | grep '^TaskCount' | awk -F\= '{print $2}' | awk -F\# '{print $1}' | xargs)
    echo "TaskCount = $TASK_COUNT (set from file: $LOCAL_FILE)"
fi

echo "Note: remember to run mwinit from Amazon VPN and then: ada credentials update --once --provider=isengard --role=admin --profile=$AWS_PROFILE --account=$AWS_ACCOUNT"

echo ""
echo "Uploading dynamic config to S3 for $TESTNAME ..."
aws s3 cp $LOCAL_FILE $S3RMT_FILE

# Check if jq is available
command -v jq >/dev/null 2>&1 || { echo "jq is required. Please install (brew install jq)" >&2; exit 1; }

echo ""
echo "Update number of tasks ..."
echo ">> aws ecs update-service --cluster=$ECS_CLUSTER --service=$ECS_SERVICE --desired-count=$TASK_COUNT"
aws ecs update-service --cluster=$ECS_CLUSTER --service=$ECS_SERVICE --desired-count=$TASK_COUNT | jq \
    '.service | {serviceName, runningCount, desiredCount}'
