'use strict';

const express = require('express');
const path = require('path');
const { createServer } = require('http');
const WebSocket = require('ws');

const app = express();
app.use(express.static(path.join(__dirname, '/public')));

let restarted = false;
const server = createServer(app);
createWebSocketServer();

const port = 6006;
server.listen(port, () => {
  console.log(`Listening on http://localhost:${port}`);
});

function createWebSocketServer() {
  const wss = new WebSocket.Server({ server });

  wss.on('connection', (ws, request) => {
    const id = `${request.socket.remoteAddress}:${request.socket.remotePort}`;
    console.log('client connected', id);
    ws.on('message', (data) => {
      console.log(`received:  ${data}`);
      try {
        const message = JSON.parse(data);
        if (message.connect) {
          const token = message.connect.token;
          const validationErrorValue = validateConnectMessage(message);
          if (validationErrorValue) {
            const errorMessage = createErrorMessage('connect_invalid_values',
              'connect message has an invalid value for a required field',
              validationErrorValue);
            ws.send(JSON.stringify(errorMessage));
          } else {
            switch (token) {
              case 'bad-token':
                ws.send(JSON.stringify(createErrorMessage('invalid_connect_token', 'auth token is not valid')));
                break;
              case 'unexpected-payload':
                ws.send(message.connect.game_id);
                break;
              case 'error':
                ws.send(JSON.stringify(createErrorMessage(message.connect.game_id, 'debug')));
                break;
              case 'abort':
                ws.send(JSON.stringify({ connected: true }));
                restartServer();
                break;
              case 'restart':
                ws.send(JSON.stringify({ connected: true }));
                restartServer(Number(message.connect.game_id));
                break;
              default:
                ws.send(JSON.stringify({ connected: true }));
                ws.send('"connected"');
            }
          }
        } else if (message.flush) {
          const flushId = message.flush;
          ws.send(JSON.stringify({ "flushed": flushId }));
        } else if (message.refresh) {
          ws.send(JSON.stringify({ debug: message }));
        } else {
          ws.send(JSON.stringify({ debug: message }));
        }
      } catch (ex) {
        console.log(ex);
        const errorMessage = createErrorMessage('invalid_message',
          'message is not an accepted type or cannot be parsed in expected format');
        ws.send(JSON.stringify(errorMessage));
      }

      function restartServer(reconnect) {
        if (restarted) {
          restarted = false;
          ws.send('"connected"');
        } else {
          restarted = true;
          if (reconnect) {
            ws.send(JSON.stringify({ reconnect }));
          }
          wss.close();
          createWebSocketServer();
        }
      }
    });
    ws.on('close', () => {
      console.log('client disconnected');
    });
  });
}

function createErrorMessage(error, message, errorField) {
  const rv = {
    error: {
      code: error,
      message,
    },
  };
  if (errorField) {
    rv.error.error_field = errorField;
  }
  return rv;
}

function validateConnectMessage(message) {
  if (!message.connect) {
    throw new Error('no "connect" field');
  }
  const missingField = ['data', 'env', 'game_id', 'session_id', 'token'].find((name) => {
    if (!message.connect[name]) {
      return name;
    }
  });
  if (missingField) {
    return missingField;
  }
  if (!['dev', 'prod'].some((s) => s == message.connect.env)) {
    return 'env';
  }
  if (message.connect.token === 'app' && !(message.connect.broadcaster_ids || []).length) {
    return 'broadcaster_ids';
  }
  if (message.connect.token === 'user' && (message.connect.broadcaster_ids || []).length) {
    return 'broadcaster_ids';
  }
}
