﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using Newtonsoft.Json.Linq;

namespace Twitch.EnhancedExperiences
{
    internal class Accessor
    {
        internal delegate DeltaOperation AccessDataFn(JObject data);
        internal delegate void AccessListFn(List<DeltaOperation> list);

        private static readonly Regex pathRx = new Regex(@"^\w+(\.\w+|\[[0-9]\])*$", RegexOptions.ECMAScript);
        private readonly object access = new object();
        private readonly Func<JObject, bool> isConnectTooLarge;
        private readonly List<DeltaOperation> list = new List<DeltaOperation>();
        private JObject data;

        internal Accessor(Func<JObject, bool> isConnectTooLarge, JObject data)
        {
            this.isConnectTooLarge = isConnectTooLarge;
            this.data = data;
        }

        internal void Access(string path, AccessDataFn fn)
        {
            // Validate the path.
            if (String.IsNullOrWhiteSpace(path))
            {
                throw new ArgumentException("path is empty", nameof(path));
            }
            else if (!pathRx.IsMatch(path))
            {
                throw new ArgumentException($"\"{path}\" is not a valid JSON path", nameof(path));
            }

            lock (access)
            {
                // Create a copy of the data.
                var data_ = (JObject)data.DeepClone();

                // Invoke the value-modifying function.
                var operation = fn(data_);
                if (operation == null)
                {
                    // The function did not modify a value.
                    return;
                }

                // Ensure the "delta" message is not too large.
                if (operation.IsDeltaTooLarge())
                {
                    throw new ArgumentException($"\"{path}\" value is too large", nameof(path));
                }

                // Ensure the "connect" message is not too large.
                if (isConnectTooLarge(data_))
                {
                    throw new ArgumentException($"\"{path}\" value is too large", nameof(path));
                }

                // Keep the updated data.
                data = data_;

                // Trim the list if possible.
                if (operation.Type == DeltaOperationType.Append)
                {
                    // If the last operation is an append to the same array, add this
                    // operation to that one if doing so does not make that delta too large.
                    var previous = list.LastOrDefault();
                    if (previous != null && previous.Type == DeltaOperationType.Append && previous.Path == operation.Path)
                    {
                        var previousValue = (JValue[])previous.Value;
                        var combinedValue = previousValue.Concat((JValue[])operation.Value).ToArray();
                        var combined = new DeltaOperation { Path = path, Type = DeltaOperationType.Append, Value = combinedValue };
                        if (!combined.IsDeltaTooLarge())
                        {
                            previous.Value = combinedValue;
                            return;
                        }
                    }
                }
                else
                {
                    // Keep only the most recent "remove" and "update" operations for a path.
                    list.RemoveAll((e) => e.Type == operation.Type && e.Path == operation.Path);

                    // Remove any "append" operations that appear before a corresponding
                    // "remove" or "update" operation for a path.  I cannot remove "update"
                    // operations that correspond to "remove" operations since those might be
                    // adding a new field and E2 will send an error message if it receives a
                    // non-corresponding "remove" operation.
                    list.RemoveAll((e) => e.Type == DeltaOperationType.Append && e.Path == operation.Path);

                    // Remove any "remove" operations that appear before a corresponding
                    // "update" operation for a path.
                    if (operation.Type == DeltaOperationType.Update)
                    {
                        list.RemoveAll((e) => e.Type == DeltaOperationType.Remove && e.Path == operation.Path);
                    }
                }

                // Add the operation to the list.
                list.Add(operation);
            }
        }

        internal void Access(AccessListFn fn)
        {
            lock (access)
            {
                fn(list);
            }
        }

        internal JObject ClearMessages()
        {
            lock (access)
            {
                list.Clear();
                return (JObject)data.DeepClone();
            }
        }
    }
}
