﻿using System;
using System.Linq;
using System.Runtime.Serialization;
using System.Threading.Tasks;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Twitch.EnhancedExperiences
{
    public partial class DataSource
    {
        public enum AbsenceBehavior { Error, None, Warning }
        public delegate void TokenRefreshFn(string token);
        public delegate bool TokenExpiredFn(TokenRefreshFn fn);

        public class Configuration
        {
            public string[] BroadcasterIds;
            public string GameId;
            public object InitialData;
            public bool IsDebugging;
            public bool IsProduction;
            public TokenExpiredFn OnTokenExpired;
            public string SessionId;
            public TimeSpan? Timeout;
            public string Token;

            public void Validate(string parameterName)
            {
                if (BroadcasterIds != null && BroadcasterIds.Any((s) => String.IsNullOrWhiteSpace(s)))
                {
                    throw new ArgumentException("BroadcasterIds contains invalid values", parameterName);
                }
                if (String.IsNullOrWhiteSpace(GameId))
                {
                    throw new ArgumentException("GameId is empty", parameterName);
                }
                ValidateData(InitialData, parameterName);
                if (SessionId != null && String.IsNullOrWhiteSpace(SessionId))
                {
                    throw new ArgumentException("SessionId is empty", parameterName);
                }
                if (Timeout.HasValue && Timeout.Value <= TimeSpan.Zero)
                {
                    throw new ArgumentException("Timeout is not a positive duration", parameterName);
                }
                if (String.IsNullOrWhiteSpace(Token))
                {
                    throw new ArgumentException("Token is invalid", parameterName);
                }
            }
        }

        internal static void ValidateData(object data, string parameterName)
        {
            // Validate the data.
            if (JToken.FromObject(data) is JObject json)
            {
                // If there is a "_metadata" field, ensure it is an object.
                if (json.TryGetValue("_metadata", out JToken metadata) && metadata.Type != JTokenType.Object)
                {
                    throw new ArgumentException("_metadata field is not an object", parameterName);
                }
            }
            else
            {
                throw new ArgumentException("data is not an object", parameterName);
            }
        }

        private void ValidateConnection()
        {
            if (connection == null)
            {
                throw new ConnectionException(isConnected: false);
            }
        }
    }

    internal partial class Connection
    {
        internal class Configuration
        {
            internal JObject InitialData { get; }
            internal DataSource.TokenExpiredFn OnTokenExpired { get; }
            internal string SessionId { get; }
            internal TimeSpan? Timeout { get; }

            private const int maximumConnectMessageSize = 100000;
            private readonly Func<JObject, JObject> createConnectMessageFn;
            private string token;

            internal Configuration(DataSource.Configuration configuration)
            {
                configuration.Validate(nameof(configuration));

                // Construct a copy of the initial data.
                InitialData = (JObject)JObject.FromObject(configuration.InitialData).DeepClone();

                // Make the "create connect message" function and copy the configuration properties.
                var broadcasterIds = configuration.BroadcasterIds?.ToArray();
                var gameId = configuration.GameId;
                var isDebugging = configuration.IsDebugging;
                var isProduction = configuration.IsProduction;
                OnTokenExpired = configuration.OnTokenExpired;
                SessionId = configuration.SessionId ?? Guid.NewGuid().ToString();
                Timeout = configuration.Timeout;
                token = configuration.Token;
                createConnectMessageFn = (JObject data) => CreateConnectMessage(gameId, isProduction, broadcasterIds, isDebugging, data);

                // Verify the initial connect message isn't too large.
                if (IsConnectTooLarge(InitialData))
                {
                    throw new ArgumentException("initial data object is too large", nameof(configuration));
                }
            }

            internal JObject CreateConnectMessage(JObject data) => createConnectMessageFn(data);

            internal bool IsConnectTooLarge(JObject data)
            {
                var message = createConnectMessageFn(data);
                return message.ToString(Formatting.None).Length > maximumConnectMessageSize;
            }

            internal async Task<bool> ProcureToken()
            {
                var source = new TaskCompletionSource<string>();
                void fn(string token)
                {
                    source.SetResult(String.IsNullOrWhiteSpace(token) ? null : token);
                }
                var result = OnTokenExpired?.Invoke(fn);
                if (result.HasValue && result.Value)
                {
                    var token_ = await source.Task;
                    if (!String.IsNullOrWhiteSpace(token_))
                    {
                        token = token_;
                        return true;
                    }
                }
                return false;
            }

            private JObject CreateConnectMessage(string gameId, bool isProduction, string[] broadcasterIds, bool isDebugging, JObject data)
            {
                var message = JObject.FromObject(new
                {
                    connect = new
                    {
                        data,
                        env = isProduction ? "prod" : "dev",
                        game_id = gameId,
                        session_id = SessionId,
                        token,
                    },
                });
                if (broadcasterIds != null)
                {
                    message["connect"]["broadcaster_ids"] = new JArray(broadcasterIds);
                }
                if (isDebugging)
                {
                    message["connect"]["debug"] = new JValue(true);
                }
                return message;
            }
        }
    }

    [Serializable]
    public class ConnectionException : Exception
    {
        public bool IsConnected { get; }

        public ConnectionException(bool isConnected, Exception innerException = null) : base(isConnected ? "already connected" : "not connected", innerException)
        {
            IsConnected = isConnected;
        }

        protected ConnectionException(SerializationInfo info, StreamingContext context) : base(info, context)
        {
        }
    }
}
